﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   アプレットマネージャが提供する機能の API の宣言
 *
 * @details
 *  本ヘッダでは、各種 shim ライブラリ開発者向けにアプレットマネージャが
 *  提供する機能の API を定義しています。
 *
 *  また、ライブラリアプレットの shim ライブラリ開発者向けには、
 *  主にライブラリアプレットの起動と終了に必要な機能を提供しています。
 *
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_Result.h>
#include <nn/applet/applet_Types.h>
#include <nn/ncm/ncm_ContentMetaId.h>


namespace nn { namespace applet {

//! @name 各種 shim ライブラリ向けの API
//! @{

//-----------------------------------------------------------------------------
/**
 * @brief   自プログラム向けの アプレットリソースユーザ ID を取得します。
 *
 * @return  自プログラム向けの AppletResourceUserId が返ります。
 *
 * @details
 *  自プログラムを一意に識別可能な アプレットリソースユーザ ID を取得します。@n
 *  返値は AppletResourceUserId 型で、以下のような用途を想定しています。
 *
 *  - クライアント側の shim ライブラリは、本 API で AppletResourceUserId を取得し、サーバープロセスに渡します。
 *  - サーバープロセスはサーバー側のインスタンスと AppletResourceUserId とを関連付けて管理します。
 *  - アプレットマネージャは、上記のサーバープロセスに対して AppletResourceUserId 単位で複数のインスタンスの活性状態をまとめて制御します。
 *
 *  AppletResourceUserId はシステム上でユニークな値であり、
 *  本 API を何度発行しても同一プログラム内であれば常に同じ
 *  AppletResourceUserId を返します。
 *
 *  また、以下のいずれの初期化も行なわれていない状態で本 API を呼び出した場合、
 *  nn::applet::AppletResourceUserId::GetInvalidId() と同じものを返します。
 *
 *  - アプリケーションとしての初期化（ nn::oe::Initialize() ）
 *  - システムアプレットとしての初期化（ nn::ae::InvokeSystemAppletMain() ）
 *  - ライブラリアプレットとしての初期化（ nn::ae::InvokeLibraryAppletMain() ）
 *
 */
AppletResourceUserId GetAppletResourceUserId() NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   呼出元アプレットのアプレットリソースユーザ ID を取得します。
 *
 * @param[out] pOutAruid    呼出元アプレットの AppletResourceUserId の格納先
 *
 * @retval  nn::applet::ResultNoCallerApplet
 * @retval  nn::applet::ResultCallerAppletIsWinding
 *
 * @details
 *  自プログラムの呼出元アプレット（アプリケーションを含む）の
 *  AppletResourceUserId を pOutAruid に格納します。
 *
 *  呼出元アプレットが存在しない場合には、
 *  nn::applet::ResultNoCallerApplet() が返ります。
 *
 *  さらに、呼出元がライブラリアプレット等で Winding のために
 *  一時的にプロセスを終了している状態の場合には、
 *  nn::applet::ResultCallerAppletIsWinding() が返ります。
 *
 *  通常、本 API はライブラリアプレットにリンクされる shim ライブラリの
 *  中から使用されることを想定しています。
 *
 */
Result GetAppletResourceUserIdOfCallerApplet(AppletResourceUserId* pOutAruid) NN_NOEXCEPT;

//! @}

//-----------------------------------------------------------------------------

}} // namespace nn::applet

