﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief APM ライブラリの API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "apm_Types.h"

namespace nn { namespace apm {

//-----------------------------------------------------------------------------
//! @name APM ライブラリの API
//! @{

/**
 * @brief       特定の性能モードで動作しているときに使用される性能を変更します。
 *
 * @param[in]   performanceMode 性能モードを示す @ref nn::apm::PerformanceMode 型の列挙値です。
 * @param[in]   performanceConfiguration 性能を示す @ref nn::apm::PerformanceConfiguration 型の列挙値です。@n
 *              performanceMode で指定した性能モードで使用可能な性能のみ指定可能です。
 *
 * @post        現在、指定した性能モードで動作している場合、性能が変更されます。@n
 *              指定した性能モードでない場合は、以降その性能モードに遷移したときに、指定した性能が使用されます。
 *
 * @details     特定の性能モードで動作しているときに使用される性能を変更します。
 */
void SetPerformanceConfiguration(nn::apm::PerformanceMode performanceMode, nn::apm::PerformanceConfiguration performanceConfiguration) NN_NOEXCEPT;

/**
 * @brief       特定の性能モードで動作しているときに使用される性能を取得します。
 *
 * @param[in]   performanceMode 性能モードを示す @ref nn::apm::PerformanceMode 型の列挙値です。
 *
 * @return      @ref nn::apm::PerformanceConfiguration 型の列挙値で性能を返します。@n
 *
 * @details     特定の性能モードで動作しているときに使用される性能を取得します。@n
 *              アプリケーションが引数で指定した性能モードに対して過去に　@nn::apm::SetPerformanceConfiguration を使用していない場合はデフォルトの性能を返します。
 */
nn::apm::PerformanceConfiguration GetPerformanceConfiguration(nn::apm::PerformanceMode performanceMode) NN_NOEXCEPT;

/**
 * @brief       現在の性能モードを取得します。
 *
 * @return      現在の性能モードです。
 *
 * @details     現在の性能モードを取得します。
 */
PerformanceMode GetPerformanceMode() NN_NOEXCEPT;

//! @}

//-----------------------------------------------------------------------------

}} // namespace nn::apm
