﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <utility>
#include <memory>
#include <mutex>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_ConditionVariable.h>

#include <nn/am/service/core/am_Applet.h>
#include <nn/am/service/core/am_ThreadImplementedApplet.h>
#include <nn/am/service/core/am_StackableApplet.h>
#include <nn/am/service/process/am_NsProcess.h>

namespace nn { namespace am { namespace service { namespace process {

/**
    @brief プロセスをメインロジックとし、process winding 可能なアプレット

    @details
     Run は次の動作を行う。

     - CreateProcess でプロセスを作成
     - プロセスの終了を待機
       - この間 RunBehindProcess を実行
     - DoReservation を呼ぶ
       - false が返ったら終了
       - true が返ったらループ

     TerminateOne は、起動中のプロセスがあれば、そのプロセスの terminate を行い、Run ループを抜ける。
*/
class WindableApplet
    : public service::core::detail::ThreadImplementedApplet<core::StackableApplet>
{
private:

    // overrides Applet
    //virtual void RequestExitOne() NN_NOEXCEPT NN_OVERRIDE final;
    virtual void TerminateOne() NN_NOEXCEPT NN_OVERRIDE final;

    // overrides ThreadImplementedApplet
    virtual Result Run() NN_NOEXCEPT NN_OVERRIDE final;

    // pure
    virtual Result CreateProcess(std::shared_ptr<NsProcess>* pOut) NN_NOEXCEPT = 0;

    // virtual: プロセス実行中の通信
    virtual void RunBehindProcess(NsProcess* pProcess) NN_NOEXCEPT;

    // virtual: winding
    virtual bool DoReservation() NN_NOEXCEPT;

protected:

    // virtual: 処理追加用(see: Run 実装)
    virtual Result Before() NN_NOEXCEPT; // Run の最初
    virtual Result After() NN_NOEXCEPT; // Run の最後(正常系のみ)
    virtual void Cleanup() NN_NOEXCEPT; // Run の最後(Before に成功した場合に必ず)

    virtual void OnProcessBegin(os::ProcessId processId) NN_NOEXCEPT; // プロセス起動前
    virtual void OnProcessEnd(os::ProcessId processId) NN_NOEXCEPT; // プロセス終了後

    // for derived
    bool IsLiving() const NN_NOEXCEPT;

    // for derived
    Result KillProcess() NN_NOEXCEPT;

    os::ProcessId GetCurrentProcessId() const NN_NOEXCEPT;

    // utility for derived
    // CleanupLock を誰かが保持されている間、Cleanup を先に進めない
private:
    class CleanupLockImpl;
public:
    typedef std::shared_ptr<CleanupLockImpl> CleanupLock;
    CleanupLock AcquireCleanupLock() NN_NOEXCEPT;

private:

    mutable os::Mutex m_Mutex{false};
    bool m_Living{true};
    std::shared_ptr<NsProcess> m_CurrentProcess;

    // ロック機構
    std::weak_ptr<CleanupLockImpl> m_WeakCleanupLock;
    os::Event m_CleaupEvent{os::EventClearMode_ManualClear};

    void KillProcessImpl(std::unique_lock<decltype(m_Mutex)> lk) NN_NOEXCEPT;
};

}}}}
