﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_MultipleWait.h>
#include <memory>
#include <nn/am/service/am_ServiceDiagnostics.h>

namespace nn { namespace am { namespace service { namespace core {

/**
    @brief アプレットの基底クラス

    @details
     以下の状態を持つ。

     - NotStarted(初期状態)
       ↓(Start() 呼び出し成功)
     - Running
       ↓(処理終了)
     - Joinable
       ↓(Join() 呼び出し)
     - Joined
*/
class Applet
    : public diagnostics::AppletBase
{
public:

    /**
        @brief destructor

        @pre NotStarted || Joined
    */
    virtual ~Applet() NN_NOEXCEPT {}

    /**
        @brief アプレットの動作を開始する

        @pre NotStarted
        @post Running (Success 時のみ)

        @details
         ライブラリアプレットなどの場合には、正しくパラメータがセットされてから呼ばれる。
    */
    virtual void Start() NN_NOEXCEPT = 0;

    /**
        @brief アプレットの動作を終了の待機を試行する(同時呼び出し可)

        @pre Running || Joinable || Joined
        @post true が返ったとき Joined
    */
    virtual bool TryJoin() NN_NOEXCEPT = 0;

    /**
        @brief TryJoin() が true を返すかもしれないことを表す MultiWaitHolder の初期化

        @pre Running || Joinable

        @details
         Join() 呼び出し前に os::FinalizeMultiWaitHolder を呼ぶこと
    */
    virtual void InitializeMultiWaitHolder(os::MultiWaitHolderType* pHolder) NN_NOEXCEPT = 0;

    /**
        @brief アプレットの動作を待機する(同時呼び出し可)

        @pre Running || Joinable || Joined
        @post Joined
    */
    void Join() NN_NOEXCEPT;

    /**
        @brief 終了命令
    */
    virtual void Terminate() NN_NOEXCEPT =0;

    /**
        @brief 結果の取得

        @pre Joined
    */
    virtual Result GetResult() NN_NOEXCEPT = 0;

#if NN_AM_SERVICE_DIAGNOSTICS_ENABLED

    virtual const char* GetAppletKindString() const NN_NOEXCEPT
    {
        return "(unknown)";
    }

#endif

};

/**
    @brief 待機用ヘルパ
*/
inline void InitializeMultiWaitHolder(os::MultiWaitHolderType* pHolder, Applet* pApplet) NN_NOEXCEPT
{
    pApplet->InitializeMultiWaitHolder(pHolder);
}

}}}}
