﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <utility>
#include <memory>
#include <nn/util/util_Optional.h>

#include <nn/applet/applet_Types.h>
#include <nn/am/service/am_IntegratedApplet.h>
#include <nn/am/service/am_Foundation.sfdl.h>
#include <nn/am/service/am_Functions.sfdl.h>
#include <nn/am/service/am_AppletProxy.h>
#include <nn/am/service/am_Storage.h>
#include <nn/am/service/am_ErrorReport.h>
#include <nn/am/service/am_ApplicationFunctions.h>
#include <nn/sf/sf_Types.h>
#include <nn/ns/ns_ApplicationLaunchInfo.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/os/os_SdkMutex.h>

namespace nn { namespace am { namespace service {

class IntegratedApplication
    : public IntegratedMainApplet
{
private:

    typedef IntegratedMainApplet Derived;

public:

    explicit IntegratedApplication(AppletSystem* pAppletSystem, os::ProcessId processId, const ns::ApplicationLaunchInfo& launchInfo) NN_NOEXCEPT;
    explicit IntegratedApplication(AppletSystem* pAppletSystem, const applet::AppletIdentityInfo& appletIdentityInfo) NN_NOEXCEPT;
    explicit IntegratedApplication(AppletSystem* pAppletSystem, ncm::ApplicationId applicationId) NN_NOEXCEPT;
    virtual ~IntegratedApplication() NN_NOEXCEPT;

    // 理由なく使用しないこと
    const ns::ApplicationControlProperty& RefControlProperty() const NN_NOEXCEPT
    {
        return m_ControlProperty;
    }

    // ControlProperty アクセッサ
    ns::Hdcp GetHdcpPolicy() const NN_NOEXCEPT
    {
        return m_ControlProperty.hdcp;
    }
    ns::PlayLogPolicy GetPlayLogPolicy() const NN_NOEXCEPT
    {
        return m_ControlProperty.playLogPolicy;
    }
    Bit32 GetSupportedLanguageFlag() const NN_NOEXCEPT
    {
        return m_ControlProperty.supportedLanguageFlag;
    }
    ns::LogoHandling GetLogoHandlingPolicy() const NN_NOEXCEPT
    {
        return m_ControlProperty.logoHandling;
    }
    ns::Screenshot GetScreenShotPolicy() const NN_NOEXCEPT
    {
        return m_ControlProperty.screenShot;
    }
    ns::VideoCapture GetVideoCapturePolicy() const NN_NOEXCEPT
    {
        return m_ControlProperty.videoCapture;
    }
    Bit64 GetSeedForPseudoDeviceId() const NN_NOEXCEPT
    {
        return m_ControlProperty.seedForPseudoDeviceId;
    }
    std::add_lvalue_reference<const char[16]>::type GetDisplayVersion() const NN_NOEXCEPT
    {
        return m_ControlProperty.displayVersion;
    }
    PlayLogQueryPolicy GetPlayLogQueryPolicy() const NN_NOEXCEPT
    {
        PlayLogQueryPolicy ret = {};
        ret.capability = m_ControlProperty.playLogQueryCapability;
        std::memcpy(&ret.queryableApplicationId, m_ControlProperty.playLogQueryableApplicationId, sizeof(ret.queryableApplicationId));
        return ret;
    }

    // LaunchProperty アクセッサ
    ncm::ApplicationId GetApplicationId() const NN_NOEXCEPT
    {
        return m_pApplicationLaunchInfo->id;
    }
    uint32_t GetApplicationVersion() const NN_NOEXCEPT
    {
        return m_pApplicationLaunchInfo->version;
    }
    ncm::StorageId GetLaunchedStorageId() const NN_NOEXCEPT
    {
        return m_pApplicationLaunchInfo->applicationStorageId;
    }
    ncm::StorageId GetPatchStorageId() const NN_NOEXCEPT
    {
        return m_pApplicationLaunchInfo->patchStorageId;
    }

    Result TerminateAndSetResult(Result result) NN_NOEXCEPT;
    void UpdateApplicationScreenShotPermission(bool isPermitted) NN_NOEXCEPT;
    Result GetDesiredLanguageImpl(settings::LanguageCode* pOut) NN_NOEXCEPT;

    virtual Result SubmitApplicationInfo() NN_NOEXCEPT NN_OVERRIDE;

    // 試遊台向け
    void SetExitTimerForQuest(uint32_t playableTime, uint32_t idleDetectionTimer) NN_NOEXCEPT
    {
        m_PlayableTimeForQeust = playableTime;
        m_IdleDetectionTimeForQuest = idleDetectionTimer;
    }
    uint32_t GetPlayableTimeForQuest() NN_NOEXCEPT
    {
        return m_PlayableTimeForQeust;
    }
    uint32_t GetIdleDetectionTimeForQuest() NN_NOEXCEPT
    {
        return m_IdleDetectionTimeForQuest;
    }

    // 自アプリケーションへの操作
    Result PushToInChannel(applet::LaunchParameterKind kind, sf::SharedPointer<IStorage> p) NN_NOEXCEPT;

    // for AppletSystem
    sf::SharedPointer<IApplicationAccessor> CreateAccessor() NN_NOEXCEPT;
    void SetApplicationLaunchRequestInfo(const applet::ApplicationLaunchRequestInfo& info) NN_NOEXCEPT;

    NN_AM_SERVICE_DIAGNOSTICS_DEFINE_APPLET_KIND_STRING("AP")

private:
    void InitializeImplForConstructors() NN_NOEXCEPT;

    // 基本情報
    applet::AppletIdentityInfo m_AppletIdentityInfo;
    applet::ApplicationLaunchRequestInfo m_ApplicationLaunchRequestInfo = {};

    // meta 情報
    os::SdkMutex m_ApplicationPropertiesMutex;
    util::optional<Result> m_pApplicationPropertiesResult;
    util::optional<ns::ApplicationLaunchInfo> m_pApplicationLaunchInfo;
    ns::ApplicationControlProperty m_ControlProperty{};

    Result EnsureApplicationProperties() NN_NOEXCEPT;
    arp::ApplicationLaunchProperty MakeArpApplicationLaunchProperty() const NN_NOEXCEPT;

    // 試遊台情報
    uint32_t m_PlayableTimeForQeust{0};
    uint32_t m_IdleDetectionTimeForQuest{0};

    // 入力
    std::shared_ptr<StorageChannel> m_Channels[3];
    static int GetIndexByLaunchParameterKind(applet::LaunchParameterKind kind) NN_NOEXCEPT;
    std::shared_ptr<StorageChannel> GetInChannel(applet::LaunchParameterKind kind) NN_NOEXCEPT;

    virtual GpuResourceGroupId GetGpuResourceGroupId() const NN_NOEXCEPT NN_OVERRIDE
    {
        return GpuResourceGroupId_Application;
    }

    ApplicationErrorReportInfo GetApplicationErrorReportInfo() const NN_NOEXCEPT;

    // overrides(option)
    virtual Result Before() NN_NOEXCEPT NN_OVERRIDE;
    virtual void Cleanup() NN_NOEXCEPT NN_OVERRIDE;

    // overrides IntegratedApplet
    virtual applet::AppletIdentityInfo GetAppletIdentityInfo() NN_NOEXCEPT NN_OVERRIDE final;
    virtual Result CreateProcessImpl(std::shared_ptr<process::NsProcess>* pOut) NN_NOEXCEPT NN_OVERRIDE final;
    virtual void BeforeActivateProxy(os::ProcessId) NN_NOEXCEPT final;
    virtual void AfterDeactivateProxy(os::ProcessId) NN_NOEXCEPT final;
    virtual std::shared_ptr<process::NsProcess> MakeAttachedProcess(os::ProcessId processId) NN_NOEXCEPT NN_OVERRIDE final;
    virtual void OnLostForeground() NN_NOEXCEPT NN_OVERRIDE final;
    virtual void OnGetForeground() NN_NOEXCEPT NN_OVERRIDE final;
    virtual void SendPlayLog( pdm::AppletEventType eventType ) NN_NOEXCEPT NN_OVERRIDE final;

    virtual bool IsScreenShotEnabled() NN_NOEXCEPT NN_OVERRIDE final;
    virtual Result SetScreenShotPermissionImpl(applet::ScreenShotPermission permission) NN_NOEXCEPT NN_OVERRIDE final;

    // overrides IntegratedApplet
    virtual AppletProxyInfo DoCreateAppletProxy(os::ProcessId processId) NN_NOEXCEPT NN_OVERRIDE;

    // for impl
    std::shared_ptr<IntegratedApplication> SharedFromThis() NN_NOEXCEPT
    {
        return std::static_pointer_cast<IntegratedApplication>(this->shared_from_this());
    }

    class ApplicationAccessorImpl;
    class ApplicationSelfProxyImpl;

    // overrieds (window)
    virtual window::WindowProperty GetInitialWindowProperty() NN_NOEXCEPT NN_OVERRIDE
    {
        auto ret = IntegratedApplet::GetInitialWindowProperty();
        ret.applicationCoreUsageRequest = CoreUsageRequest::Exclusive;
        ret.handlesPerformanceModeChanged = true;
        ret.gpuTimeSliceInForeground = GetGpuTimeSliceOf(GpuTimeKind_FgApp);
        ret.gpuTimeSliceInBackground = GetGpuTimeSliceOf(GpuTimeKind_BgApp);
        ret.gpuTimeSlicePriority = 255;
        ret.foregroundMode = window::ForegroundMode::All;
        ret.determinesHomeButtonLongPressingTime = true;
        ret.isVrModeDualScreenSupported = true;
        ret.pWirelessPriorityModeRequest = WirelessPriorityMode_OptimizedForBluetooth;
        return ret;
    }

private:

    os::Mutex m_GameCardSequenceMutex{false};
    int64_t m_InitialGameCardSequenceNumber;
    bool IsGameCardStateValidForThis() NN_NOEXCEPT;
    void CheckGameCardState() NN_NOEXCEPT;

    // overrides IntegratedApplet
    virtual void OnGameCardStateChanged() NN_NOEXCEPT NN_OVERRIDE;

    Result m_ApplicationResult{ResultSuccess()};

    util::optional<os::ProcessId> m_pProcessIdForFloating;

private:

    // マルチプログラムアプリケーション関連
    class SubProgramJumpInfo
    {
    private:

        mutable os::SdkMutex m_Mutex;
        util::optional<ncm::ApplicationId> m_pNextApplicationId;
        util::optional<uint8_t> m_pNextProgramIndex;

        int32_t m_CurrentProgramIndex = -1;
        int32_t m_PreviousProgramIndex = -1;

    public:

        std::pair<ncm::ApplicationId, uint8_t> GetNext(ncm::ApplicationId defaultApplicationId) NN_NOEXCEPT;
        void OnExecuteProgram(uint8_t newProgramIndex) NN_NOEXCEPT;
        int32_t GetPreviousProgramIndex() const NN_NOEXCEPT;
        int32_t GetCurrentProgramIndex() const NN_NOEXCEPT;

        void ChangeNextApplicationId(ncm::ApplicationId applicationId) NN_NOEXCEPT;
        void ChangeNextProgramIndex(uint8_t programIndex) NN_NOEXCEPT;

    };
    SubProgramJumpInfo m_SubProgramJumpInfo;

    Result ExecuteNextProgram() NN_NOEXCEPT;

private:

    // 権利関連
    virtual void OnRightsActivated(os::Tick expirationTick) NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnRightsDeactivated() NN_NOEXCEPT NN_OVERRIDE;
    virtual void OnRightsAvailableTimeChanged(os::Tick expirationTick) NN_NOEXCEPT NN_OVERRIDE;

};

}}}
