﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/baas/account_BaasOperator.h>
#include <nn/account/profile/account_ProfileStorage.h>
#include <nn/account/user/account_UserReference.h>
#include <nn/account/user/account_UserRegistry.h>
#include <nn/account/account_Result.h>

#include <nn/nn_Result.h>
#include <nn/sf/sf_Out.h>
#include <nn/sf/sf_Buffers.h>
#include <nn/time/time_StandardUserSystemClock.h>

namespace nn { namespace account { namespace profile {

class ProfileEditorImpl
{
private:
    ProfileStorage& m_ProfileStorage;
    baas::BaasOperator* m_pBaasOp;
    user::UserRef m_UserRef;
    Uid m_Uid;

public:
    ProfileEditorImpl(const Uid& uid, ProfileStorage& storage) NN_NOEXCEPT
        : m_ProfileStorage(storage)
        , m_pBaasOp(nullptr)
        , m_Uid(uid)
    {
        // Pending なユーザー用
    }
    ProfileEditorImpl(user::UserRef&& user, ProfileStorage& storage, baas::BaasOperator& baasOp) NN_NOEXCEPT
        : m_ProfileStorage(storage)
        , m_pBaasOp(&baasOp)
        , m_UserRef(std::move(user))
        , m_Uid(m_UserRef)
    {
        // Registered なユーザー用
    }
    nn::Result StoreWithImage(const ProfileBase& profileBase, const UserData& userData, const sf::InBuffer& pImage) NN_NOEXCEPT
    {
        auto base = profileBase;
        base.author = m_Uid;

        time::PosixTime t;
        NN_RESULT_DO(time::StandardUserSystemClock::GetCurrentTime(&t));
        base.timeStamp = (t.value > 0 ? static_cast<uint64_t>(t.value) : 0);

        {
            auto lock = m_ProfileStorage.GetStorageRef().AcquireWriterLock();
            NN_RESULT_DO(m_ProfileStorage.Update(m_Uid, base, userData, pImage.GetPointerUnsafe(), pImage.GetSize()));
            NN_RESULT_DO(m_ProfileStorage.GetStorageRef().Commit());
        }

        if (m_pBaasOp != nullptr)
        {
            // 次回 NSA ログイン時にプロフィール同期が行われるようにする
            m_pBaasOp->InvalidateUserResourceCache(m_Uid);
        }
        NN_RESULT_SUCCESS;
    }
    nn::Result Store(const ProfileBase& profileBase, const UserData& userData) NN_NOEXCEPT
    {
        return StoreWithImage(profileBase, userData, sf::InBuffer(nullptr, 0u));
    }
    nn::Result Get(sf::Out<ProfileBase> pOutProfileBase, sf::Out<UserData> pOutUserData) const NN_NOEXCEPT
    {
        return m_ProfileStorage.GetProfile(pOutProfileBase.GetPointer(), pOutUserData.GetPointer(), m_Uid);
    }
    nn::Result GetBase(sf::Out<ProfileBase> pOutProfileBase) const NN_NOEXCEPT
    {
        return m_ProfileStorage.GetProfile(pOutProfileBase.GetPointer(), nullptr, m_Uid);
    }
    nn::Result GetImageSize(sf::Out<uint32_t> pOutSize) const NN_NOEXCEPT
    {
        size_t size;
        NN_RESULT_DO(m_ProfileStorage.GetImageSize(&size, m_Uid));
        NN_ABORT_UNLESS(
            (size & ~0xFFFFFFFFull) == 0,
            "[nn::account] -----------------------------------------------\n"
            "  ABORT: Too large (> 0xFFFFFFFF bytes) profile image\n");
        *pOutSize = static_cast<uint32_t>(size);
        NN_RESULT_SUCCESS;
    }
    nn::Result LoadImage(sf::Out<uint32_t> pOutActualSize, const sf::OutBuffer& pOutImage) const NN_NOEXCEPT
    {
        size_t actualSize;
        NN_RESULT_DO(m_ProfileStorage.LoadImage(&actualSize, pOutImage.GetPointerUnsafe(), pOutImage.GetSize(), m_Uid));
        NN_ABORT_UNLESS(
            (actualSize & ~0xFFFFFFFFull) == 0,
            "[nn::account] -----------------------------------------------\n"
            "  ABORT: Too large (> 0xFFFFFFFF bytes) profile image\n");
        *pOutActualSize = static_cast<uint32_t>(actualSize);
        NN_RESULT_SUCCESS;
    }
};

}}} // ~namespace nn::account::impl
