﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/detail/account_Execution.h>
#include <nn/account/ndas/account_AuthenticationCache.h>
#include <nn/account/ndas/account_NdasDriver.h>
#include <nn/account/account_RuntimeResource.h>

#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace account { namespace ndas {

class NdasOperator
{
private:
    const NdasDriver m_NdasDriver;

    Result HandleDeviceAuthenticationError(const Result& result) NN_NOEXCEPT;
    Result AuthenticateServiceImpl(uint64_t clientId, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result ReAuthenticateServiceImpl(uint64_t clientId, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;

public:
    explicit NdasOperator(
        ApplicationAuthenticationCache& appAuthCache) NN_NOEXCEPT
        : m_NdasDriver(appAuthCache)
    {
    }

    /* -----------------------------------------------------------------------------------------
        サービス全体の認証
    */
    Result AuthenticateServiceForApplications(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
    {
        return AuthenticateServiceImpl(ClientIdForApplications, resource, pCancellable);
    }
    Result ReAuthenticateServiceForApplications(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
    {
        return ReAuthenticateServiceImpl(ClientIdForApplications, resource, pCancellable);
    }
    Result LoadServiceAuthenticationTokenCacheForApplications(size_t* pOutSizeActual, char* buffer, size_t bufferSize) const NN_NOEXCEPT
    {
        return m_NdasDriver.LoadDeviceAuthenticationCache(pOutSizeActual, buffer, bufferSize, ClientIdForApplications);
    }

    Result AuthenticateServiceForNintendoAccount(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
    {
        return AuthenticateServiceImpl(ClientIdForNintendoAccount, resource, pCancellable);
    }
    Result LoadServiceAuthenticationTokenCacheForNintendoAccount(size_t* pOutSizeActual, char* buffer, size_t bufferSize) const NN_NOEXCEPT
    {
        return m_NdasDriver.LoadDeviceAuthenticationCache(pOutSizeActual, buffer, bufferSize, ClientIdForNintendoAccount);
    }

    /* -----------------------------------------------------------------------------------------
        アプリケーション個別の認証
    */
    Result AuthenticateApplication(const detail::ApplicationInfo& appInfo, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    Result ReAuthenticateApplication(const detail::ApplicationInfo& appInfo, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT;
    bool IsApplicationAuthenticated(const detail::ApplicationInfo& appInfo) const NN_NOEXCEPT
    {
        return m_NdasDriver.IsApplicationAuthenticationCacheAvailable(appInfo);
    }
    Result LoadApplicationAuthenticationTokenCache(size_t* pOutSizeActual, char* buffer, size_t bufferSize, const detail::ApplicationInfo& appInfo) const NN_NOEXCEPT
    {
        return m_NdasDriver.LoadApplicationAuthenticationCache(pOutSizeActual, buffer, bufferSize, appInfo);
    }
    Result GetApplicationMetaCache(ApplicationMeta* pOutMeta, const detail::ApplicationInfo& appInfo) const NN_NOEXCEPT
    {
        return m_NdasDriver.GetApplicationMetaCache(pOutMeta, appInfo);
    }
};

class DeviceReAuthenticationTask
    : public detail::Executable<ExecutionResource>
{
private:
    NdasOperator& m_NdasOp;
protected:
    virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT final NN_OVERRIDE
    {
        return m_NdasOp.ReAuthenticateServiceForApplications(resource, this);
    }
public:
    explicit DeviceReAuthenticationTask(NdasOperator& ndasOp) NN_NOEXCEPT
        : m_NdasOp(ndasOp)
    {
    }
};

class ApplicationReAuthenticationTask
    : public detail::Executable<ExecutionResource>
{
private:
    NdasOperator& m_NdasOp;
    const detail::ApplicationInfo m_AppInfo;

protected:
    virtual Result ExecuteImpl(const ExecutionResource& resource) NN_NOEXCEPT final NN_OVERRIDE
    {
        return m_NdasOp.ReAuthenticateApplication(m_AppInfo, resource, this);
    }
public:
    explicit ApplicationReAuthenticationTask(const detail::ApplicationInfo& appInfo, NdasOperator& ndasOp) NN_NOEXCEPT
        : m_NdasOp(ndasOp)
        , m_AppInfo(appInfo)
    {
    }
};

}}} // ~namespace nn::account::ndas
