﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/baas/account_BaasOperator.h>
#include <nn/account/detail/account_InternalTypes.h>
#include <nn/account/nas/account_NasOperator.h>
#include <nn/account/ndas/account_NdasOperator.h>
#include <nn/account/account_Types.h>
#include <nn/account/account_TypesForSystemServices.h>

#include <new>
#include <utility>

#include <nn/os/os_SdkMutex.h>
#include <nn/util/util_ScopeExit.h>

namespace nn {
namespace account {
namespace nas {

class NasSessionPoolBase
{
public:
    static const size_t SessionCountMax = 8;

    struct Validation
    {
        int tag;
        Uid uid;
    };

private:
    os::SdkMutex m_Lock;
    struct Session
    {

        detail::Uuid sessionId;
        Validation validation;
        uint32_t count;
        uintptr_t ptr;
    } m_Sessions[SessionCountMax];

    const Session* FindUnsafe(const detail::Uuid& sessionId) const NN_NOEXCEPT;
    Session* FindUnsafe(const detail::Uuid& sessionId, const Validation& validation) NN_NOEXCEPT;

protected:
    NasSessionPoolBase() NN_NOEXCEPT;
    Result Register(detail::Uuid* pOutSessionId, uintptr_t ptr, const Validation& validation) NN_NOEXCEPT;
    Result Acquire(uintptr_t* pOutPtr, const detail::Uuid& sessionId, const Validation& validation) NN_NOEXCEPT;
    uintptr_t Release(const detail::Uuid& sessionId, const Validation& validation) NN_NOEXCEPT;
};

template <typename Allocator>
class NasSessionPool
    : NasSessionPoolBase
{
private:
    enum Tag
    {
        Tag_NasApplicationAuthorization,
        Tag_NasGuestLogin,
        Tag_NasFloatingRegistration,
    };

    Allocator& m_Allocator;

public:
    explicit NasSessionPool(Allocator& allocator) NN_NOEXCEPT;

    /* ---------------------------------------------------------
        NasApplicationAuthorization
    */
    template <typename IdentityType>
    Result CreateNasApplicationAuthorizationSession(detail::Uuid* pOutSessionId, NasApplicationAuthorizationProcedure** pOutPtr, const Uid& uid, IdentityType&& id, NasOperator& nasOp) NN_NOEXCEPT;
    Result AcquireNasApplicationAuthorizationSession(NasApplicationAuthorizationProcedure** pOutPtr, const Uid& uid, const detail::Uuid& sessionId) NN_NOEXCEPT;
    void ReleaseNasApplicationAuthorizationSession(const Uid& uid, const detail::Uuid& sessionId) NN_NOEXCEPT;

    /* ---------------------------------------------------------
        BaasGuestLogin
    */
    Result CreateNasGuestLoginSession(detail::Uuid* pOutSessionId, NasGuestLoginProcedure** pOutPtr, const detail::ApplicationInfo& appInfo, ndas::NdasOperator& ndasOp, baas::BaasOperator& baasOp, NasOperator& nasOp) NN_NOEXCEPT;
    Result AcquireNasGuestLoginSession(NasGuestLoginProcedure** pOutPtr, const detail::Uuid& sessionId) NN_NOEXCEPT;
    void ReleaseNasGuestLoginSession(const detail::Uuid& sessionId) NN_NOEXCEPT;

    /* ---------------------------------------------------------
        BaasFloatingRegistration
    */
    Result CreateNasFloatingRegistrationSession(detail::Uuid* pOutSessionId, NasFloatingRegistrationProcedure** pOutPtr, ndas::NdasOperator& ndasOp, baas::BaasOperator& baasOp, NasOperator& nasOp) NN_NOEXCEPT;
    Result AcquireNasFloatingRegistrationSession(NasFloatingRegistrationProcedure** pOutPtr, const detail::Uuid& sessionId) NN_NOEXCEPT;
    void ReleaseNasFloatingRegistrationSession(const detail::Uuid& sessionId) NN_NOEXCEPT;
};

} // ~nn::account::nas
}
}

/* ---------------------------------------------------------------------------------------------
    実装
*/
namespace nn {
namespace account {
namespace nas {

template <typename Allocator>
inline NasSessionPool<Allocator>::NasSessionPool(Allocator& allocator) NN_NOEXCEPT
    : m_Allocator(allocator)
{
}

template <typename Allocator> template <typename IdentityType>
inline Result NasSessionPool<Allocator>::CreateNasApplicationAuthorizationSession(
    detail::Uuid* pOutSessionId, NasApplicationAuthorizationProcedure** pOutPtr, const Uid& uid, IdentityType&& id, NasOperator& nasOp) NN_NOEXCEPT
{
    auto memory = m_Allocator.Allocate(sizeof(NasApplicationAuthorizationProcedure));
    NN_RESULT_THROW_UNLESS(memory != nullptr, ResultNasAuthorizationSessionAllocationError());

    auto* p = new(memory) NasApplicationAuthorizationProcedure(uid, std::forward<IdentityType>(id), nasOp);
    bool requiresDeallocation = true;
    NN_UTIL_SCOPE_EXIT
    {
        if (requiresDeallocation)
        {
            p->~NasApplicationAuthorizationProcedure();
            m_Allocator.Deallocate(p, sizeof(*p));
        }
    };

    Validation validation = {};
    validation.tag = Tag_NasApplicationAuthorization;
    validation.uid = uid;
    NN_RESULT_DO(this->Register(pOutSessionId, reinterpret_cast<uintptr_t>(p), validation));
    requiresDeallocation = false;
    *pOutPtr = p;
    NN_RESULT_SUCCESS;
}

template <typename Allocator>
inline Result NasSessionPool<Allocator>::AcquireNasApplicationAuthorizationSession(NasApplicationAuthorizationProcedure** pOutPtr, const Uid& uid, const detail::Uuid& sessionId) NN_NOEXCEPT
{
    Validation validation = {};
    validation.tag = Tag_NasApplicationAuthorization;
    validation.uid = uid;
    uintptr_t ptr;
    NN_RESULT_DO(this->Acquire(&ptr, sessionId, validation));
    *pOutPtr = reinterpret_cast<decltype(*pOutPtr)>(ptr);
    NN_RESULT_SUCCESS;
}

template <typename Allocator>
inline void NasSessionPool<Allocator>::ReleaseNasApplicationAuthorizationSession(const Uid& uid, const detail::Uuid& sessionId) NN_NOEXCEPT
{
    Validation validation = {};
    validation.tag = Tag_NasApplicationAuthorization;
    validation.uid = uid;
    auto ptr = this->Release(sessionId, validation);
    if (ptr != 0u)
    {
        auto* p = reinterpret_cast<NasApplicationAuthorizationProcedure*>(ptr);
        p->~NasApplicationAuthorizationProcedure();
        m_Allocator.Deallocate(p, sizeof(*p));
    }
}

template <typename Allocator>
inline Result NasSessionPool<Allocator>::CreateNasGuestLoginSession(detail::Uuid* pOutSessionId, NasGuestLoginProcedure** pOutPtr, const detail::ApplicationInfo& appInfo, ndas::NdasOperator& ndasOp, baas::BaasOperator& baasOp, NasOperator& nasOp) NN_NOEXCEPT
{
    auto memory = m_Allocator.Allocate(sizeof(NasGuestLoginProcedure));
    NN_RESULT_THROW_UNLESS(memory != nullptr, ResultNasAuthorizationSessionAllocationError());

    auto* p = new(memory) NasGuestLoginProcedure(appInfo, ndasOp, baasOp, nasOp);
    bool requiresDeallocation = true;
    NN_UTIL_SCOPE_EXIT
    {
        if (requiresDeallocation)
        {
            p->~NasGuestLoginProcedure();
            m_Allocator.Deallocate(p, sizeof(*p));
        }
    };

    Validation validation = {};
    validation.tag = Tag_NasGuestLogin;
    validation.uid = InvalidUid;
    NN_RESULT_DO(this->Register(pOutSessionId, reinterpret_cast<uintptr_t>(p), validation));
    requiresDeallocation = false;
    *pOutPtr = p;
    NN_RESULT_SUCCESS;
}

template <typename Allocator>
inline Result NasSessionPool<Allocator>::AcquireNasGuestLoginSession(NasGuestLoginProcedure** pOutPtr, const detail::Uuid& sessionId) NN_NOEXCEPT
{
    Validation validation = {};
    validation.tag = Tag_NasGuestLogin;
    validation.uid = InvalidUid;
    uintptr_t ptr;
    NN_RESULT_DO(this->Acquire(&ptr, sessionId, validation));
    *pOutPtr = reinterpret_cast<decltype(*pOutPtr)>(ptr);
    NN_RESULT_SUCCESS;
}

template <typename Allocator>
inline void NasSessionPool<Allocator>::ReleaseNasGuestLoginSession(const detail::Uuid& sessionId) NN_NOEXCEPT
{
    Validation validation = {};
    validation.tag = Tag_NasGuestLogin;
    validation.uid = InvalidUid;
    auto ptr = this->Release(sessionId, validation);
    if (ptr != 0u)
    {
        auto* p = reinterpret_cast<NasGuestLoginProcedure*>(ptr);
        p->~NasGuestLoginProcedure();
        m_Allocator.Deallocate(p, sizeof(*p));
    }
}

template <typename Allocator>
inline Result NasSessionPool<Allocator>::CreateNasFloatingRegistrationSession(detail::Uuid* pOutSessionId, NasFloatingRegistrationProcedure** pOutPtr, ndas::NdasOperator& ndasOp, baas::BaasOperator& baasOp, NasOperator& nasOp) NN_NOEXCEPT
{
    auto memory = m_Allocator.Allocate(sizeof(NasFloatingRegistrationProcedure));
    NN_RESULT_THROW_UNLESS(memory != nullptr, ResultNasAuthorizationSessionAllocationError());

    auto* p = new(memory) NasFloatingRegistrationProcedure(ndasOp, baasOp, nasOp);
    bool requiresDeallocation = true;
    NN_UTIL_SCOPE_EXIT
    {
        if (requiresDeallocation)
        {
            p->~NasFloatingRegistrationProcedure();
            m_Allocator.Deallocate(p, sizeof(*p));
        }
    };

    Validation validation = {};
    validation.tag = Tag_NasFloatingRegistration;
    validation.uid = InvalidUid;
    NN_RESULT_DO(this->Register(pOutSessionId, reinterpret_cast<uintptr_t>(p), validation));
    requiresDeallocation = false;
    *pOutPtr = p;
    NN_RESULT_SUCCESS;
}

template <typename Allocator>
inline Result NasSessionPool<Allocator>::AcquireNasFloatingRegistrationSession(NasFloatingRegistrationProcedure** pOutPtr, const detail::Uuid& sessionId) NN_NOEXCEPT
{
    Validation validation = {};
    validation.tag = Tag_NasFloatingRegistration;
    validation.uid = InvalidUid;
    uintptr_t ptr;
    NN_RESULT_DO(this->Acquire(&ptr, sessionId, validation));
    *pOutPtr = reinterpret_cast<decltype(*pOutPtr)>(ptr);
    NN_RESULT_SUCCESS;
}

template <typename Allocator>
inline void NasSessionPool<Allocator>::ReleaseNasFloatingRegistrationSession(const detail::Uuid& sessionId) NN_NOEXCEPT
{
    Validation validation = {};
    validation.tag = Tag_NasFloatingRegistration;
    validation.uid = InvalidUid;
    auto ptr = this->Release(sessionId, validation);
    if (ptr != 0u)
    {
        auto* p = reinterpret_cast<NasFloatingRegistrationProcedure*>(ptr);
        p->~NasFloatingRegistrationProcedure();
        m_Allocator.Deallocate(p, sizeof(*p));
    }
}

} // ~nn::account::nas
}
}
