﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/detail/account_InternalTypes.h>
#include <nn/account/detail/account_InternalConfig.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SdkMutex.h>

namespace nn { namespace account { namespace detail {

class AbstractFileSystem;
Uuid GenerateUuid(const AbstractFileSystem& fs) NN_NOEXCEPT;

}}} // ~namespace nn::account::detail

namespace nn { namespace account { namespace detail {

struct SystemSaveDataPolicy
{
private:
    mutable os::SdkMutex m_Mutex;

public:
    struct Policy
    {
        typedef SystemSaveDataPolicy SaveData;
    };
    const char* const VolumeName = SaveDataVolumeName;

public:
    template <typename FileSystem>
    Result Mount() const NN_NOEXCEPT;
    template <typename FileSystem>
    nn::Result Commit() const NN_NOEXCEPT;

    template <typename FileSystem>
    Uuid GenerateUuid() const NN_NOEXCEPT;
};

}}} // ~namespace nn::account::detail

#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace account { namespace detail {

template <typename FileSystem>
inline Result SystemSaveDataPolicy::Mount() const NN_NOEXCEPT
{
    return FileSystem::MountSave(VolumeName, SaveDataId, SaveDataSize, SaveDataJournalSize, SaveDataFlags);
}

template <typename FileSystem>
inline nn::Result SystemSaveDataPolicy::Commit() const NN_NOEXCEPT
{
    return  FileSystem::Commit(VolumeName);
}

template <typename FileSystem>
inline Uuid SystemSaveDataPolicy::GenerateUuid() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);
    // ID 生成コンテキスト用ボリュームのマウント
    NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::MountSave(
        SaveDataVolumeNameForIdGeneration, SaveDataIdForIdGeneration,
        SaveDataSizeForIdGeneration, SaveDataJournalSizeForIdGeneration, SaveDataFlagsForIdGeneration));
    NN_UTIL_SCOPE_EXIT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::Commit(SaveDataVolumeNameForIdGeneration));
        FileSystem::Unmount(SaveDataVolumeNameForIdGeneration);
    };
    // UUID 生成
    return nn::account::detail::GenerateUuid(FileSystem());
}

}}} // ~namespace nn::account::detail
