﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_NintendoAccountTypes.h>
#include <nn/account/account_TypesForSystemServices.h>
#include <nn/account/detail/account_OAuthProcedureBase.h>

namespace nn {
namespace account {
namespace http {
class IOAuthProcedure;
} // ~namespace nn::account::http
namespace nas {
class IOAuthProcedureForExternalNsa;
class IOAuthProcedureForNintendoAccountLinkage;
} // ~namesapce nn::account::nas

}} // ~namespace nn::account

namespace nn {
namespace account {

/**
    @brief アカウントシステムでニンテンドーアカウント連携のための手続きを実施するためのクラスです。
    @details
        アカウントシステムでニンテンドーアカウント連携のための手続きを実施するためのクラスです。

        このクラスでは、OAuth 2.0で必要となるユーザーの認証や認可を、Webブラウザー等のユーザーエージェント経由で要求するために必要なパラメータを提供します。
        また認証や認可の結果を含むレスポンス文字列の入力を受け付けることで、認証や認可の結果をアカウントシステムに反映します。
*/
class NintendoAccountLinkageProcedure final
    : detail::OAuthProcedureBase
{
    NN_DISALLOW_COPY(NintendoAccountLinkageProcedure);

protected:
    nas::IOAuthProcedureForNintendoAccountLinkage* GetPtr() const NN_NOEXCEPT
    {
        return reinterpret_cast<nas::IOAuthProcedureForNintendoAccountLinkage*>(OAuthProcedureBase::GetPtr());
    }

public:
    /**
        @brief ニンテンドーアカウント連携のための手続きオブジェクトを作成します。

        @post
            - このクラスの機能を使用不可能

        @details
            ニンテンドーアカウント連携のための手続きオブジェクトを作成します。
            この状態のオブジェクトは実際の手続きのためのオブジェクトを出力するAPIへの入力として使用します。
    */
    NintendoAccountLinkageProcedure() NN_NOEXCEPT;
    virtual ~NintendoAccountLinkageProcedure() NN_NOEXCEPT {}

    /**
        @brief ユーザーの認証や認可のリクエストのためのパラメータを取得します。

        @param[out] pOutRequestUrl  認証や認可を開始するためのリクエストURL
        @param[out] pOutCallbackUri 認証や認可の完了を示すコールバックURI
        @param[in] theme リクエスト時に表示されるWebページのテーマ

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理に成功しました。
            }
        @endretresult

        @pre
            - オブジェクトが実際の手続きと紐づいている
            - pOutRequestUrl != nullptr
            - pOutCallbackUri != nullptr

        @details
            ユーザーの認証や認可を得るためのパラメータを取得します。
            このパラメータには次の要素があります。
            - リクエストURL
                - Webブラウザー等のユーザーエージェントでこのURLに対してリクエストを行うと、ユーザーに認証や認可の手続きを求めることができます。
            - コールバックURI
                - 認証や認可の手続きが行われると、接続先のサーバーはここで得られるURIに対するリダイレクトを行います。
                  ただし実際にリダイレクトされるURIには、URIフラグメントやクエリパラメータとして認証や認可の結果が含まれることがあります。
    */
    Result GetRequest(RequestUrl* pOutRequestUrl, CallbackUri* pOutCallbackUri, NintendoAccountAuthorizationPageTheme theme) NN_NOEXCEPT;

    using OAuthProcedureBase::GetRequest;
    using OAuthProcedureBase::ApplyResponseAsync;
    using OAuthProcedureBase::Suspend;

    /**
        @brief ニンテンドーアカウント連携の結果、ユーザーのネットワークサービスアカウントが置き換わったかを取得します。

        @param[out] pOut ユーザーのネットワークサービスアカウントが置き換わったかを格納するアドレス

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理に成功しました。
            }
        @endretresult

        @pre
            - ApplyResponseAsync() による非同期処理が成功している

        @details
            ニンテンドーアカウント連携の結果、ユーザーのネットワークサービスアカウントが置き換わったかを取得します。
    */
    Result IsNetworkServiceAccountReplaced(bool* pOut) NN_NOEXCEPT;

    // 内部向け
    NintendoAccountLinkageProcedure(const Uid& user, nas::IOAuthProcedureForNintendoAccountLinkage* ptr) NN_NOEXCEPT;
    NintendoAccountLinkageProcedure(NintendoAccountLinkageProcedure&& rhs) NN_NOEXCEPT;
    NintendoAccountLinkageProcedure& operator=(NintendoAccountLinkageProcedure&& rhs) NN_NOEXCEPT;
    NintendoAccountLinkageProcedure& Swap(NintendoAccountLinkageProcedure& rhs) NN_NOEXCEPT;
};

/**
    @brief アカウントシステムでニンテンドーアカウントとの連携状態を更新する手続きを実施するためのクラスです。
    @details
        アカウントシステムでニンテンドーアカウントとの連携状態を更新する手続きを実施するためのクラスです。
*/
using NintendoAccountLinkageStateUpdateProcedure = detail::SimpleOAuthProcedureBase;

/**
    @brief アカウントシステムでNNIDをニンテンドーアカウントと連携する手続きを実施するためのクラスです。
    @details
        アカウントシステムでNNIDをニンテンドーアカウントと連携する手続きを実施するためのクラスです。
*/
using NintendoAccountNnidLinkageProcedure = detail::SimpleOAuthProcedureBase;

/**
    @brief アカウントシステムでアプリケーションに対するニンテンドーアカウントの認可取得の手続きを実施するためのクラスです。
    @details
        アカウントシステムでアプリケーションに対するニンテンドーアカウントの認可取得の手続きを実施するためのクラスです。
*/
class NintendoAccountApplicationAuthorizationProcedure final
    : detail::OAuthProcedureBase
{
    NN_DISALLOW_COPY(NintendoAccountApplicationAuthorizationProcedure);

public:
    /**
        @brief アプリケーションに対するニンテンドーアカウントの認可取得の手続きオブジェクトを作成します。

        @post
            - このクラスの機能を使用不可能

        @details
            アプリケーションに対するニンテンドーアカウントの認可取得の手続きオブジェクトを作成します。
            この状態のオブジェクトは実際の手続きのためのオブジェクトを出力するAPIへの入力として使用します。
    */
    NintendoAccountApplicationAuthorizationProcedure() NN_NOEXCEPT;
    virtual ~NintendoAccountApplicationAuthorizationProcedure() NN_NOEXCEPT {};

    using OAuthProcedureBase::PrepareAsync;
    using OAuthProcedureBase::GetRequest;
    using OAuthProcedureBase::ApplyResponse;

    // 内部向け
    NintendoAccountApplicationAuthorizationProcedure(const Uid& user, http::IOAuthProcedure* ptr) NN_NOEXCEPT;
    NintendoAccountApplicationAuthorizationProcedure(NintendoAccountApplicationAuthorizationProcedure&& rhs) NN_NOEXCEPT;
    NintendoAccountApplicationAuthorizationProcedure& operator=(NintendoAccountApplicationAuthorizationProcedure&& rhs) NN_NOEXCEPT;
    NintendoAccountApplicationAuthorizationProcedure& Swap(NintendoAccountApplicationAuthorizationProcedure& rhs) NN_NOEXCEPT;
};

/**
    @brief アカウントシステム外部のネットワークサービスアカウントを一時的に利用可能にする手続きを実施するためのクラスです。
    @details
        アカウントシステム外部のネットワークサービスアカウントを一時的に利用可能にする手続きを実施するためのクラスです。
*/
class ExternalNetworkServiceAccountIntroducingProcedure final
    : detail::OAuthProcedureBase
{
    NN_DISALLOW_COPY(ExternalNetworkServiceAccountIntroducingProcedure);

protected:
    nas::IOAuthProcedureForExternalNsa* GetPtr() const NN_NOEXCEPT
    {
        return reinterpret_cast<nas::IOAuthProcedureForExternalNsa*>(OAuthProcedureBase::GetPtr());
    }

public:
    /**
        @brief アカウントシステム外部のネットワークサービスアカウントを一時的に利用可能にするための手続きオブジェクトを作成します。

        @post
            - このクラスの機能を使用不可能

        @details
            アカウントシステム外部のネットワークサービスアカウントを一時的に利用可能にするための手続きオブジェクトを作成します。
            この状態のオブジェクトは実際の手続きオブジェクトを出力するAPIに指定して使用します。
    */
    ExternalNetworkServiceAccountIntroducingProcedure() NN_NOEXCEPT;
    virtual ~ExternalNetworkServiceAccountIntroducingProcedure() NN_NOEXCEPT {};

    using OAuthProcedureBase::PrepareAsync;
    using OAuthProcedureBase::GetRequest;
    using OAuthProcedureBase::ApplyResponseAsync;

    /**
        @brief ネットワークサービスアカウントのニックネームを取得します。

        @param[out] pOut ニックネームの格納先

        @retresult
            @handleresult{
                nn::ResultSuccess,
                成功しました。
            }
        @endretresult

        @pre
            - *this が有効である
            - pOut != nullptr
        @post
            - pOut->name[0, NicknameBytesMax] にひとつ以上の '\0' が存在

        @details
            ネットワークサービスアカウントのニックネームを取得します。
            このニックネームは nn::account::GetNickname() で取得できるものと互換性を持ちます。
    */
    Result GetNickname(Nickname* pOut) const NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントのプロフィール画像を取得します。

        @param[out] pOutActualSize プロフィール画像の実際の大きさ (バイト数) の格納先
        @param[out] outImage プロフィール画像の格納先
        @param[in] bufferSize outImage に書き込み可能な大きさ (バイト数)

        @retresult
            @handleresult{
                nn::ResultSuccess,
                成功しました。
            }
        @endretresult

        @pre
            - *this が有効である
            - pOutActualSize != nullptr
        @post
            - *pOutActualSize が画像データの実際の大きさ (バイト数) を保持
            - outImage != nullptr のとき、 outImage[0, min(*pOutActualSize, bufferSize)) が画像データを保持
            - outImage の保持する画像データをJPEGとして解釈可能

        @details
            ネットワークサービスアカウントのプロフィール画像を取得します。
            このプロフィール画像の性質は nn::account::LoadProfileImage() で取得できるものと互換性を持ちます。
    */
    Result GetProfileImage(size_t* pOutActualSize, void* outImage, size_t bufferSize) const NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウント ID を取得します。

        @param[out] pOutId ネットワークサービスアカウント ID

        @retresult
            @handleresult{
                nn::ResultSuccess,
                成功しました。
            }
        @endretresult

        @pre
            - *this が有効である
            - pOutId != nullptr
        @post
            - *pOutId が有効なネットワークサービスアカウント ID を保持

        @details
            ネットワークサービスアカウント ID を取得します。
    */
    Result GetNetworkServiceAccountId(NetworkServiceAccountId* pOutId) const NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントと連携しているニンテンドーアカウントの内部 ID を取得します。

        @param[out] pOutId ニンテンドーアカウントの内部 ID

        @retresult
            @handleresult{
                nn::ResultSuccess,
                成功しました。
            }
        @endretresult

        @pre
            - *this が有効である
            - pOutId != nullptr
        @post
            - *pOutId が有効なニンテンドーアカウントの内部 ID を保持

        @details
            ネットワークサービスアカウントと連携しているニンテンドーアカウントの内部 ID を取得します。
    */
    Result GetLinkedNintendoAccountId(NintendoAccountId* pOutId) const NN_NOEXCEPT;

    // 内部向け
    explicit ExternalNetworkServiceAccountIntroducingProcedure(nas::IOAuthProcedureForExternalNsa* ptr) NN_NOEXCEPT;
    ExternalNetworkServiceAccountIntroducingProcedure(ExternalNetworkServiceAccountIntroducingProcedure&& rhs) NN_NOEXCEPT;
    ExternalNetworkServiceAccountIntroducingProcedure& operator=(ExternalNetworkServiceAccountIntroducingProcedure&& rhs) NN_NOEXCEPT;
    ExternalNetworkServiceAccountIntroducingProcedure& Swap(ExternalNetworkServiceAccountIntroducingProcedure& rhs) NN_NOEXCEPT;
};

}
} // ~namespace nn::account
