﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/** @file
    @brief このファイルは本体システム外部のネットワークサービスアカウントの利用に対応するアプリケーションが、その利用のために呼ぶAPIを宣言します。
 */

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>

namespace nn {
namespace account {

class ExternalNetworkServiceAccountInfo;

namespace baas {
class IGuestLoginRequest;
} // ~namespace nn::account::baas

}
}

namespace nn { namespace account {

//! @name 本体システム外部のネットワークサービスアカウントに関する機能
//! @{

/**
    @brief 有効な ExternalNetworkServiceAccountInfo オブジェクトの作成に必要なバッファの大きさ(バイト数)
*/
const size_t RequiredBufferSizeForExternalNetworkServiceAccountInfo = 34 * 4096;

/**
    @brief 本体システム外部のネットワークサービスアカウントをアプリケーション内で一時的に使用可能にします。

    @param[out] pOutInfo 使用可能となったネットワークサービスアカウントの情報を保持するオブジェクト
    @param[out] workBuffer 使用可能となったネットワークサービスアカウントの情報の保持に必要なワークメモリ。
    @param[in] workBufferSize workBufferの大きさ(バイト数)

    @retresult
        @handleresult{
            nn::ResultSuccess,
            成功しました。
        }
        @handleresult{
            nn::account::ResultCancelledByUser,
            ユーザー操作によって操作が中止されました。
            この結果をエラーとして扱わないでください。
        }
    @endretresult

    @pre
        - pOutInfo != nullptr
        - workBuffer != nullptr
        - reinterpret_cast<uintptr_t>(workBuffer) % nn::os::MemoryPageSize == 0
        - workBufferSize >= nn::account::RequiredBufferSizeForExternalNetworkServiceAccountInfo
        - workBufferSize % nn::os::MemoryPageSize == 0
    @post
        - *pOutInfo が有効である
        - *pOutInfo が有効な間、 workBuffer から workBufferSize バイトの範囲にアクセスできない

    @details
        この関数を呼ぶと本体システムのUIが表示され、ユーザー操作を経て、本体システムに登録されていないネットワークサービスアカウントを一時的に利用可能な状態にします。
        本関数で使用可能になるネットワークサービスアカウントは本体システムに登録されず、その情報は *pOutInfo の破棄とともに破棄されます。

        *pOutInfo から参照できる情報は本関数の呼び出し中に取得された情報であり、これは自動的に更新されません。
        IDトークンの期限切れなどによって情報を再取得する必要が生じた場合は、本関数を再度呼びなおしてください。

        この関数で使用可能になるネットワークサービスアカウントは、関数が呼ばれた時点で本体システムに登録されていないネットワークサービスアカウントであることが保証されます。
        一方で本関数呼び出し後に登録されたものや、別の ExternalNetworkServiceAccountInfo オブジェクトが保持するものとは重複する可能性があります。

        ワークメモリとして指定するメモリ領域は、 ExternalNetworkServiceAccountInfo オブジェクトが有効な情報を保持している期間はアクセスできません。

        この関数は本体システムのUIを表示するために、呼び出し元のスレッドをブロックします。
*/
Result IntroduceExternalNetworkServiceAccount(ExternalNetworkServiceAccountInfo *pOutInfo, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;

//! @}

/**
    @brief 本体システム外部のネットワークサービスアカウントの情報を保持するためのクラス
    @details
        IntroduceExternalNetworkServiceAccount() によって取得する本体システム外部のネットワークサービスアカウントの情報を保持します。
        このオブジェクトから取得可能な情報は本体システムに登録されておらず、 *pOutInfo の破棄とともに破棄されます。

        ExternalNetworkServiceAccountInfo オブジェクトは次の状態をもちます。
        - 無効
            - オブジェクトはいかなるネットワークサービスアカウントとも関連づきません。
            - この状態では、デストラクタ以外を呼び出すことはできません。
        - 有効
            - このオブジェクトは特定のネットワークサービスアカウントと関連づきます。
            - この状態ではすべてのAPIを利用することができます。
 */
class ExternalNetworkServiceAccountInfo
{
private:
    baas::IGuestLoginRequest* m_Ptr;

public:
    /**
        @brief 無効なインスタンスを作成するデフォルトコンストラクタ

        @post
            - *this は無効

        @details
            無効なインスタンスを作成します。
            この状態では、デストラクタ以外のいかなる API も呼び出すことができません。
    */
    ExternalNetworkServiceAccountInfo() NN_NOEXCEPT;
    ~ExternalNetworkServiceAccountInfo() NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントのニックネームを取得します。

        @param[out] pOut ニックネームの格納先

        @retresult
            @handleresult{
                nn::ResultSuccess,
                成功しました。
            }
        @endretresult

        @pre
            - *this が有効である
            - pOut != nullptr
        @post
            - pOut->name[0, NicknameBytesMax] にひとつ以上の '\0' が存在

        @details
            ネットワークサービスアカウントのニックネームを取得します。
            このニックネームは nn::account::GetNickname() で取得できるものと互換性を持ちます。
    */
    Result GetNickname(Nickname* pOut) const NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントのプロフィール画像を取得します。

        @param[out] pOutActualSize プロフィール画像の実際の大きさ (バイト数) の格納先
        @param[out] outImage プロフィール画像の格納先
        @param[in] bufferSize outImage に書き込み可能な大きさ (バイト数)

        @retresult
            @handleresult{
                nn::ResultSuccess,
                成功しました。
            }
        @endretresult

        @pre
            - *this が有効である
            - pOutActualSize != nullptr
        @post
            - *pOutActualSize が画像データの実際の大きさ (バイト数) を保持
            - outImage != nullptr のとき、 outImage[0, min(*pOutActualSize, bufferSize)) が画像データを保持
            - outImage の保持する画像データをJPEGとして解釈可能

        @details
            ネットワークサービスアカウントのプロフィール画像を取得します。
            このプロフィール画像の性質は nn::account::LoadProfileImage() で取得できるものと互換性を持ちます。
    */
    Result GetProfileImage(size_t* pOutActualSize, void* outImage, size_t bufferSize) const NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントIDを取得します。

        @param[out] pOutId ネットワークサービスアカウントID

        @retresult
            @handleresult{
                nn::ResultSuccess,
                成功しました。
            }
        @endretresult

        @pre
            - *this が有効である
            - pOutId != nullptr
        @post
            - *pOutId が有効なネットワークサービスアカウントIDを保持

        @details
            ネットワークサービスアカウントIDを取得します。
    */
    Result GetNetworkServiceAccountId(NetworkServiceAccountId* pOutId) const NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントのIDトークンのキャッシュを取得します。

        @param[out] pOutActualSize IDトークンの実際の長さ (終端文字含まず)
        @param[out] buffer IDトークンの格納先
        @param[in]  bufferSize buffer に書き込み可能な大きさ (バイト数)

        @retresult
            @handleresult{
                nn::ResultSuccess,
                IDトークンキャッシュの取得に成功しました。
            }
            @handleresult{
                nn::account::ResultTokenCacheUnavailable,
                IDトークンキャッシュを利用できません。
                *this を取得した IntroduceExternalNetworkServiceAccount() の呼び出しから十分な時間が経過しています。
            }
        @endretresult

        @pre
            - *this が有効である
            - pOutActualSize != nullptr
            - buffer != nullptr
            - bufferSize >= nn::account::NetworkServiceAccountIdTokenLengthMax
            - IsNetworkServiceAccountAvailable() が true を返す
        @post
            - *pOutActualSize がIDトークンの終端文字を含めない実際の長さを保持
            - buffer[0, *pOutActualSize) の範囲のASCII列を有効なIDトークンとして利用可能
            - *pOutActualSize < bufferSize のときに限りIDトークンは '\0' で終端される

        @details
            ネットワークサービスアカウントのIDトークンのキャッシュを取得します。
            IDトークンはURLエンコードされた文字列として取得され、その最大長は終端文字を含めず nn::account::NetworkServiceAccountIdTokenLengthMax です。

            本関数で取得可能なIDトークンには有効期限があります。
            失効までの猶予は IntroduceExternalNetworkServiceAccount() 呼び出しから30分かそれ以上であり、この期間内に使用されることを期待しています。
            原則的にIDトークンを使用する都度、 IntroduceExternalNetworkServiceAccount() 並びに本関数を使用してトークンを再取得してください。
            失効したIDトークンをサーバーに送信すると、サーバーでの検証に失敗する場合があります。

            本関数で取得可能なIDトークンは呼び出し元のアプリケーション固有の情報を含んでいるため、他のアプリケーションと共有しないでください。
            また、この関数で取得可能なIDトークンはセーブデータやオンラインストレージなどの記憶領域に保存しないでください。
    */
    Result LoadNetworkServiceAccountIdTokenCache(size_t* pOutActualSize, char* buffer, size_t bufferSize) const NN_NOEXCEPT;

    explicit ExternalNetworkServiceAccountInfo(baas::IGuestLoginRequest* ptr) NN_NOEXCEPT;
    ExternalNetworkServiceAccountInfo(ExternalNetworkServiceAccountInfo&& rhs) NN_NOEXCEPT;
    ExternalNetworkServiceAccountInfo& operator=(ExternalNetworkServiceAccountInfo&& rhs) NN_NOEXCEPT;
    void Swap(ExternalNetworkServiceAccountInfo& rhs) NN_NOEXCEPT;
};

}
}

