﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Reflection;
using System.Diagnostics;

namespace Nintendo.ServiceFramework
{
    public enum InOutType
    {
        In,
        Out,
    }

    public static class TypeUtility
    {
        public static IEnumerable<MethodInfo> GetAllInterfaceMethods(this Type t)
        {
            return new[] { t }.Concat(t.GetInterfaces()).SelectMany(e => e.GetMethods());
        }

        public static Type GetInheritedOutmostInterfaceImpl(this Type t)
        {
            if (!(t.GetInterfaces().Count() > 0))
            {
                throw new WrongImplementationException();
            }

            Type outMost = null;
            HashSet<Type> current = new HashSet<Type>();
            foreach (var i in t.GetInterfaces())
            {
                var given = new HashSet<Type>(i.GetInterfaces());
                given.Add(i);

                if (given.IsSupersetOf(current))
                {
                    // より多くのインターフェースを継承している方を選択する。
                    outMost = i;
                    current = given;
                }
                else if (!given.IsSubsetOf(current))
                {
                    // nn.sf.IServiceObject を継承した 2 つのインターフェースの継承関係が、
                    // どちらかを包含するものでなければ、それは多重継承である。
                    return null;
                }
            }
            return outMost;
        }

        public static bool IsSingleInheritedInterface(this Type t)
        {
            return (GetInheritedOutmostInterfaceImpl(t) != null);
        }

        public static Type GetInheritedOutmostInterface(this Type t)
        {
            var it = GetInheritedOutmostInterfaceImpl(t);
            if (it == null)
            {
                throw new WrongImplementationException();
            }
            return it;
        }

        public static IEnumerable<Type> GetBaseTypes(this Type t)
        {
            while (true)
            {
                t = t.BaseType;
                yield return t;
                if (t == typeof(object))
                {
                    yield break;
                }
            }
        }

        public static Type GetSfOutElementType(this Type t)
        {
            if (!(t.IsClass && t.IsGenericType && t.GetGenericTypeDefinition() == typeof(nn.sf.Out<>)))
            {
                return null;
            }
            return t.GenericTypeArguments[0];
        }

        public static bool IsSfOutParameterType(this Type t)
        {
            return t.IsGenericType && t.GetGenericTypeDefinition() == typeof(nn.sf.Out<>);
        }

        public static InOutType GetInOutType(this Type t)
        {
            if (t.IsSfOutParameterType())
            {
                return InOutType.Out;
            }
            else
            {
                return InOutType.In;
            }
        }

        public static Type GetSfParameterElementType(this Type t)
        {
            if (IsSfOutParameterType(t))
            {
                return t.GenericTypeArguments[0];
            }
            else
            {
                return t;
            }
        }

        public static IEnumerable<MethodInfo> SortByMethodId(this IEnumerable<MethodInfo> methods)
        {
            var ret = methods.ToArray();
            Array.Sort(ret, (x, y) =>
            {
                var xIsInIServiceObject = x.DeclaringType == typeof(nn.sf.IServiceObject);
                var yIsInIServiceObject = y.DeclaringType == typeof(nn.sf.IServiceObject);
                if (xIsInIServiceObject)
                {
                    if (yIsInIServiceObject)
                    {
                        return 0;
                    }
                    else
                    {
                        return -1;
                    }
                }
                else
                {
                    if (yIsInIServiceObject)
                    {
                        return 1;
                    }
                    else
                    {
                        return MethodIdAttribute.GetId(x).Value - MethodIdAttribute.GetId(y).Value;
                    }
                }
            });
            return ret;
        }
    }
}
