﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Reflection;

namespace Nintendo.ServiceFramework
{
    /// <summary>
    /// サービスオブジェクトインターフェイスの ID を指定するための属性
    /// </summary>
    /// <remarks>
    /// 必須属性ではない。
    /// インターフェイスをメソッド引数として取る場合など、特定の機能を使用する場合に必要となる。
    /// </remarks>
    [Serializable]
    [AttributeUsage(AttributeTargets.Interface, AllowMultiple = false)]
    public class InterfaceIdAttribute : Attribute
    {
        /// <summary>
        /// サービスオブジェクトインターフェイスの ID
        /// </summary>
        private int Id { get; set; }

        /// <summary>
        /// ID を指定するコンストラクタ
        /// </summary>
        /// <param name="id">サービスオブジェクトインターフェイスの ID</param>
        public InterfaceIdAttribute(int id)
        {
            this.Id = id;
        }

        /// <summary>
        /// (内部用) 型から、インターフェイス ID の値を取得するためのユーティリティ
        /// </summary>
        /// <param name="t">対象の型</param>
        /// <returns>インターフェイス ID</returns>
        internal static int? GetId(Type t)
        {
            var a = t.GetCustomAttribute<InterfaceIdAttribute>();
            if (a == null)
            {
                return null;
            }
            return a.Id;
        }
    }

    /// <summary>
    /// サービスオブジェクトインターフェイスのメソッドの ID を指定するための属性
    /// </summary>
    [Serializable]
    [AttributeUsage(AttributeTargets.Method, AllowMultiple = false)]
    public class MethodIdAttribute : Attribute
    {
        /// <summary>
        /// サービスオブジェクトインターフェイスのメソッドの ID
        /// </summary>
        private int Id { get; set; }

        /// <summary>
        /// ID を指定するコンストラクタ
        /// </summary>
        /// <param name="id">サービスオブジェクトインターフェイスのメソッドの ID</param>
        public MethodIdAttribute(int id)
        {
            this.Id = id;
        }

        /// <summary>
        /// (内部用) メソッドメンバから、メソッド ID の値を取得するためのユーティリティ
        /// </summary>
        /// <param name="t">対象のメソッドメンバ</param>
        /// <returns>メソッド ID</returns>
        internal static int? GetId(MethodInfo m)
        {
            var a = m.GetCustomAttribute<MethodIdAttribute>();
            if (a == null)
            {
                return null;
            }
            return a.Id;
        }
    }

    /// <summary>
    /// サービスオブジェクトインターフェイス・またはメソッドを cmif 対応させるかどうかを指定する属性
    /// </summary>
    /// <remarks>
    /// 必須属性ではなく、省略時には true として扱われる。
    /// </remarks>
    [Serializable]
    [AttributeUsage(AttributeTargets.Interface | AttributeTargets.Method, AllowMultiple = false)]
    public class UseCmifAttribute : Attribute
    {
        private bool UseCmif { get; set; }

        /// <summary>
        /// サービスオブジェクトインターフェイスを cmif 対応させるかどうかを指定するコンストラクタ
        /// </summary>
        /// <param name="useCmif">サービスオブジェクトインターフェイスを cmif 対応させるかどうか</param>
        public UseCmifAttribute(bool useCmif)
        {
            this.UseCmif = useCmif;
        }

        /// <summary>
        /// (内部用)
        /// </summary>
        /// <param name="t"></param>
        /// <returns></returns>
        internal static bool GetUseCmif(Type t)
        {
            var a = t.GetCustomAttribute<UseCmifAttribute>();
            if (a == null)
            {
                return true;
            }
            return a.UseCmif;
        }

        /// <summary>
        /// (内部用)
        /// </summary>
        /// <param name="m"></param>
        /// <returns></returns>
        internal static bool GetUseCmif(MethodInfo m)
        {
            var a = m.GetCustomAttribute<UseCmifAttribute>();
            if (a == null)
            {
                return true;
            }
            return a.UseCmif;
        }
    }

    /// <summary>
    /// サービスオブジェクトインターフェイスを impl 対応させるかどうかを指定する属性 (内部用)
    /// </summary>
    /// <remarks>
    /// 必須属性ではなく、省略時には true として扱われる。
    /// 通常、ユーザは指定する必要はない。
    /// </remarks>
    [Serializable]
    [AttributeUsage(AttributeTargets.Interface, AllowMultiple = false)]
    public class UseImplAttribute : Attribute
    {
        private bool UseImpl { get; set; }

        /// <summary>
        /// サービスオブジェクトインターフェイスを cmif 対応させるかどうかを指定するコンストラクタ
        /// </summary>
        /// <param name="useCmif">サービスオブジェクトインターフェイスを cmif 対応させるかどうか</param>
        public UseImplAttribute(bool useCmif)
        {
            this.UseImpl = useCmif;
        }

        /// <summary>
        /// (内部用)
        /// </summary>
        /// <param name="t"></param>
        /// <returns></returns>
        internal static bool GetUseImpl(Type t)
        {
            var a = t.GetCustomAttribute<UseImplAttribute>();
            if (a == null)
            {
                return true;
            }
            return a.UseImpl;
        }
    }

    [Serializable]
    [AttributeUsage(AttributeTargets.Field, AllowMultiple = false)]
    public class FixedArrayAttribute : Attribute
    {
        private int Length { get; set; }

        public FixedArrayAttribute(int length)
        {
            this.Length = length;
        }

        internal static int? GetLength(FieldInfo field)
        {
            var a = field.GetCustomAttribute<FixedArrayAttribute>();
            if (a == null)
            {
                return null;
            }
            return a.Length;
        }
    }

    [Serializable]
    [AttributeUsage(AttributeTargets.Struct, AllowMultiple = false)]
    public class ExternalStructAttribute : Attribute
    {
        private int Size { get; set; }
        private int Alignment { get; set; }

        public ExternalStructAttribute(int size, int alignment)
        {
            this.Size = size;
            this.Alignment = alignment;
        }

        internal static SizeAndAlignment GetSizeAndAlignment(Type t)
        {
            var a = t.GetCustomAttribute<ExternalStructAttribute>();
            if (a == null)
            {
                return null;
            }
            return new SizeAndAlignment
            {
                Size = a.Size,
                Alignment = a.Alignment,
            };
        }
    }

    [Serializable]
    [AttributeUsage(AttributeTargets.Parameter, AllowMultiple = false)]
    public class MoveHandleAttribute : Attribute
    {
    }

    [Serializable]
    [AttributeUsage(AttributeTargets.Struct | AttributeTargets.Parameter, AllowMultiple = false)]
    public class LargeDataAttribute : Attribute
    {
        private bool Value { get; set; }

        public LargeDataAttribute()
            : this(true)
        {
        }

        public LargeDataAttribute(bool value)
        {
            this.Value = value;
        }

        internal static bool IsLargeDataParameter(ParameterInfo p)
        {
            var t = p.ParameterType.GetSfParameterElementType();
            if (!(t.GetSfEntity() is SfValueType))
            {
                return false;
            }
            {
                // パラメータへの直接指定を優先
                var a = p.GetCustomAttribute<LargeDataAttribute>();
                if (a != null)
                {
                    return a.Value;
                }
            }
            {
                // パラメータへの明示指定がない場合には、型への指定
                var a = t.GetCustomAttribute<LargeDataAttribute>();
                if (a != null)
                {
                    return a.Value;
                }
            }
            // 指定がない場合のデフォルトは false
            return false;
        }

        internal static bool IsSpecifiedLargeDataParameter(ParameterInfo p)
        {
            return false
                || p.GetCustomAttribute<LargeDataAttribute>() != null
                || p.ParameterType.GetSfParameterElementType().GetCustomAttribute<LargeDataAttribute>() != null;
        }
    }

    [Serializable]
    [AttributeUsage(AttributeTargets.Parameter, AllowMultiple = false)]
    public class BufferParameterAttribute : Attribute
    {
        private bool Value { get; set; }

        public BufferParameterAttribute(bool value)
        {
            this.Value = value;
        }

        public static bool IsBufferParameter(ParameterInfo p)
        {
            var a = p.GetCustomAttribute<BufferParameterAttribute>();
            if (a != null)
            {
                // パラメータへの直接指定を優先
                return a.Value;
            }
            var t = p.ParameterType.GetSfParameterElementType();
            var e = t.GetSfEntity();
            if (e is SfNnBuffer)
            {
                return true;
            }
            return LargeDataAttribute.IsLargeDataParameter(p);
        }

        internal static BufferTransferInternalDirection GetBufferTransferInternalDirection(ParameterInfo p)
        {
            var t = p.ParameterType.GetSfParameterElementType();
            var e = t.GetSfEntity();
            if (e is SfNnBuffer)
            {
                return t.GetCustomAttribute<BufferTransferInternalDirectionAttribute>().Direction;
            }
            else
            {
                switch (p.ParameterType.GetInOutType())
                {
                    case InOutType.In: return BufferTransferInternalDirection.In;
                    case InOutType.Out: return BufferTransferInternalDirection.Out;
                    default: throw new WrongImplementationException();
                }
            }
        }
    }
}

namespace Nintendo.ServiceFramework.Hipc
{
    [Serializable]
    [AttributeUsage(AttributeTargets.Parameter, AllowMultiple = false)]
    public class ProcessIdAttribute : Attribute
    {
    }

    public enum BufferTransferMode
    {
        PointerCopy,
        MapAlias,
        AutoSelect,
    }

    [Serializable]
    [AttributeUsage(AttributeTargets.Parameter | AttributeTargets.Struct | AttributeTargets.Class, AllowMultiple = false)]
    public class BufferTransferModeAttribute : Attribute
    {
        private BufferTransferMode TransferMode { get; set; }

        public BufferTransferModeAttribute(BufferTransferMode transferMode)
        {
            this.TransferMode = transferMode;
        }

        internal static BufferTransferMode? GetBufferTransferMode(ParameterInfo p)
        {
            {
                // パラメータに明示指定のものを優先
                var a = p.GetCustomAttribute<BufferTransferModeAttribute>();
                if (a != null)
                {
                    return a.TransferMode;
                }
            }
            {
                // 型に明示指定のものを優先
                var t = p.ParameterType.GetSfParameterElementType();
                var a = t.GetCustomAttribute<BufferTransferModeAttribute>();
                if (a != null)
                {
                    return a.TransferMode;
                }
            }
            // BufferParameterAttribute.IsBufferParameter であればポインタ転送
            if (BufferParameterAttribute.IsBufferParameter(p))
            {
                return BufferTransferMode.PointerCopy;
            }
            return null;
        }
    }

    [Serializable]
    [AttributeUsage(AttributeTargets.Parameter | AttributeTargets.Struct | AttributeTargets.Class, AllowMultiple = false)]
    public class AutoSelectBufferAttribute : BufferTransferModeAttribute
    {
        public AutoSelectBufferAttribute()
            : base(BufferTransferMode.AutoSelect)
        {
        }
    }

    public enum MapTransferSecurity
    {
        Secure = 0,
        NonSecure = 1,
        NonDevice = 3,
    }

    [Serializable]
    [AttributeUsage(AttributeTargets.Parameter, AllowMultiple = false)]
    public class MapTransferSecurityAttribute : Attribute
    {
        private MapTransferSecurity MapTransferSecurity { get; set; }

        public MapTransferSecurityAttribute(MapTransferSecurity mapTransferSecurity)
        {
            this.MapTransferSecurity = mapTransferSecurity;
        }

        internal static MapTransferSecurity GetMapTransferSecurity(ParameterInfo p)
        {
            var a = p.GetCustomAttributes<MapTransferSecurityAttribute>().ToArray();
            switch (a.Length)
            {
                case 0: return MapTransferSecurity.Secure;
                case 1: return a[0].MapTransferSecurity;
                default: throw new WrongImplementationException();
            }
        }
    }

    [Serializable]
    [AttributeUsage(AttributeTargets.Parameter, AllowMultiple = false)]
    public class NonSecureMapTransferAttribute : MapTransferSecurityAttribute
    {
        public NonSecureMapTransferAttribute()
            : base(MapTransferSecurity.NonSecure)
        {
        }
    }
}
