﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Reflection;
using System.Threading.Tasks;

namespace Nintendo.ServiceFramework.CppCode
{
    public static class EntityCppCodeEmitterUtility
    {
        public static IEnumerable<string> GetParameterStringList(this MethodInfo method, bool withType, bool friendly)
        {
            return from p in method.GetParameters()
                   let name = p.Name
                   let type = p.ParameterType
                   let elementType = TypeUtility.GetSfParameterElementType(type)
                   let entity = elementType.GetSfEntity()
                   select withType
                       ? entity.GetParameterString(name, TypeUtility.GetInOutType(type), friendly)
                       : entity.GetForwardString(name, TypeUtility.GetInOutType(type), friendly);
        }

        public static string GetMacroFunctionString(this MethodInfo method, string prefix, bool withType, bool friendly, string postfix)
        {
            return string.Format(@"{0}_{1}_{2}({3}{4}{5}){6}",
                prefix,
                method.GetReturnMacroString(),
                method.GetParameterCountString(),
                method.Name,
                method.GetReturnTypeParameterString(),
                method.GetParametersString(withType, friendly),
                postfix);
        }

        public static string GetMacroFunctionString(this MethodInfo method, string prefix, bool withType, bool friendly)
        {
            return GetMacroFunctionString(method, prefix, withType, friendly, string.Empty);
        }

        #region GetMacroFunctionString 実装

        private static string GetReturnMacroString(this MethodInfo method)
        {
            var returnType = method.ReturnType;
            if (returnType == typeof(void))
            {
                return @"VOID";
            }
            else if (returnType == typeof(nn.Result))
            {
                return @"RESULT";
            }
            else
            {
                return @"VALUE";
            }
        }

        private static string GetParameterCountString(this MethodInfo method)
        {
            return method.GetParameters().Length == 0 ? @"0" : @"N";
        }

        private static string GetReturnTypeParameterString(this MethodInfo method)
        {
            if (method.ReturnType == typeof(void) || method.ReturnType == typeof(nn.Result))
            {
                return string.Empty;
            }
            else
            {
                return string.Format(@", ({0})", method.ReturnType.GetSfEntity().GetCppTypeString(false));
            }
        }

        private static string GetParametersString(this MethodInfo method, bool withType, bool friendly)
        {
            var parameters = method.GetParameters();
            if (parameters.Length == 0)
            {
                return string.Empty;
            }
            else
            {
                var args = GetParameterStringList(method, withType, friendly);
                return string.Format(@", ({0})", string.Join(@", ", args.ToArray()));
            }
        }

        #endregion

        public static string GetReturnTypeString(this MethodInfo method, bool friendly)
        {
            if (method.ReturnType == typeof(void))
            {
                return @"void";
            }
            else
            {
                return method.ReturnType.GetSfEntity().GetCppTypeString(friendly);
            }
        }
    }
}
