﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading;
using System.Diagnostics;
using Nintendo.InitializeSdev;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;

namespace Nintendo.SystemUpdateSdev
{
    public partial class SdevSystemUpdate
    {
        private const int RunOnTargetInterval = 4;
        private const int ResetInterval = 20;

        private string GetVersionString()
        {
            if (Argument.targetFirmwareVersion == null)
            {
                return string.Empty;
            }
            else
            {
                return Argument.targetFirmwareVersion;
            }
        }

        private ExitStatus StopTargetManager()
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Stop NintendoTargetManager.");
            if (ProcessAccessor.StopTargetManager() != Nintendo.InitializeSdev.ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "NintendoTargetManager Stopped.");

            return ExitStatus.Success;
        }

        private ExitStatus StopTargetManagerForLast()
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Stop NintendoTargetManager .");
            if (ProcessAccessor.StopTargetManagerForLast() != Nintendo.InitializeSdev.ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "NintendoTargetManager Stopped.");

            return ExitStatus.Success;
        }

        private Dictionary<TargetSpecifier.SdevVersion, string> sdevVersionDictionary = new Dictionary<TargetSpecifier.SdevVersion, string>()
        {
            {TargetSpecifier.SdevVersion.SDEV_1_5, "1.5"},
            {TargetSpecifier.SdevVersion.SDEV_1_6, "1.6"},
            {TargetSpecifier.SdevVersion.SDEV_1_6_G, "1.6+G"},
            {TargetSpecifier.SdevVersion.SDEV_1_8, "1.8"},
            {TargetSpecifier.SdevVersion.SDEV_PRE_MP1, "preMP1"},
            {TargetSpecifier.SdevVersion.SDEV_PRE_MP2, "preMP2"},
            {TargetSpecifier.SdevVersion.SDEV_MP, "MP"},
            {TargetSpecifier.SdevVersion.SDEV_NONE, ""},
            {TargetSpecifier.SdevVersion.SDEV_ERROR, ""},
        };

        private ExitStatus CheckSdevVersion(out TargetSpecifier.SdevVersion sdevVersion, TargetInnerClass targetKey)
        {
            TargetSpecifier.SdevVersion version = TargetSpecifier.GetSdevVersion(targetKey);
            sdevVersion = version;
            switch (version)
            {
                case TargetSpecifier.SdevVersion.SDEV_1_5:
                case TargetSpecifier.SdevVersion.SDEV_1_6:
                case TargetSpecifier.SdevVersion.SDEV_1_6_G:
                case TargetSpecifier.SdevVersion.SDEV_1_8:
                case TargetSpecifier.SdevVersion.SDEV_PRE_MP1:
                case TargetSpecifier.SdevVersion.SDEV_PRE_MP2:
                case TargetSpecifier.SdevVersion.SDEV_MP:
                    LOG.LogLine(LOG_LEVEL.LOG_INFO2, "SDEV version is {0}", sdevVersionDictionary[version]);
                    break;
                case TargetSpecifier.SdevVersion.SDEV_ERROR:
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "SDEV version check failed.");
                    return ExitStatus.Failure;
                default:
                    LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Unsupport target.");
                    return ExitStatus.Failure;
            }
            int sleepInterval = 15000;
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Sleep {0} msec.", sleepInterval);
            Thread.Sleep(15000);
            return ExitStatus.Success;
        }

        public ExitStatus ValidateSdevVersion(FirmwareResourceSpecifier firmwareResource)
        {
            // Prod もしくは ProdWithLog の場合はK5鍵の機種のみ利用可能
            if(!firmwareResource.ValidateSetVersion())
            {
                return ExitStatus.Failure;
            }

            return ExitStatus.Success;
        }

#if false
        public void RegisterTargetIfTargetTypeIsIp(TargetInnerClass targetKey, string targetManagerPath)
        {
            if(targetKey.Type != TargetInnerClass.TargetType.TargetInner_SdevIp_Force)
            {
                return;
            }

            TargetManagerAccess.RegisterTarget(targetKey.Value, targetManagerPath);
        }
#endif

        public ExitStatus UpdateHostBridgeFirmware(TargetInnerClass targetKey, string hostbridgeFirmwarePath, bool recovery = false)
        {
            ExitStatus eStatus = ExitStatus.Failure;

            if (Argument.disableHostBridgeFirmwareUpdate == true)
            {
                LOG.LogLine(LOG_LEVEL.LOG_INFO2, "skipped HostBridge FW update.");
                return ExitStatus.Success;
            }

#if false
            bool specifiedIpAddress = (targetKey.Type == TargetInnerClass.TargetType.TargetInner_SdevIp_Force);
#endif
            string targetIp = TargetSpecifier.GetTargetIp(targetKey);
            if (targetIp == null)
            {
                LOG.LogLine(LOG_LEVEL.LOG_WARN, "not found target ({0}).", targetKey.Value);
                return ExitStatus.Failure;
            }


            //handle.Release();

            LOG.LogLine("Update HostBridge FW.");

            // Call UpdateHostBridge
            List<string> args1;
            if (!recovery)
            {
                args1 = new List<string>() { "update", "--target", targetIp, "--image", hostbridgeFirmwarePath };
            }
            else
            {
                args1 = new List<string>() { "update", "--target", targetIp, "--image", hostbridgeFirmwarePath, "--recovery" };
            }
#if false
            if (specifiedIpAddress == false)
#endif
            {
                args1.Add("--waitboot");
            }
            if (Argument.Verbose)
            {
                args1.Add("--verbose");
            }
            List<string> argsList = new List<string>();
            argsList.AddRange(args1);
#if !CALL_UPDATE_HOSTBRIDGE_EXE
            ProcessAccessor.DumpProcessArgumentList("UpdateHostBridge", argsList);

            int retVar = Nintendo.UpdateHostBridge.CommandInterface.Main(argsList.ToArray());
            if (retVar != 0)
            {
                retVar = Nintendo.UpdateHostBridge.CommandInterface.Main(argsList.ToArray());
            }

#if false
            if (specifiedIpAddress == true)
            {
                Thread.Sleep(30000);
            }
#endif

            eStatus = retVar == 0 ? ExitStatus.Success : ExitStatus.Failure;
#else
            const string updateHostBridgePath = "UpdateHostBridge.exe";
            int exitCode = -1;

            ProcessAccessor.DumpProcessArgumentList("UpdateHostBridge", argsList);

            exitCode = ProcessAccessor.DoProcess(updateHostBridgePath, argsList, OutputDataReceived);
            if (exitCode != 0)
            {
                exitCode = ProcessAccessor.DoProcess(updateHostBridgePath, argsList, OutputDataReceived);
                if (exitCode != 0)
                {
                    goto quit;
                }
            }

            eStatus = ExitStatus.Success;
        quit:
#endif
            if (eStatus == ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_INFO2, "UpdateHostBridge passed.");
            }
            else
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "UpdateHostBridge failed.");
            }
            return eStatus;
        }

        public ExitStatus UpdateRecoveryHostBridgeFirmware(TargetInnerClass targetKey, string hostbridgeRecoveryFirmwarePath)
        {
            ExitStatus eStatus = ExitStatus.Failure;

            // 復旧モード用FWのアップデート処理
            bool required;
            eStatus = IsUpdateHostBridgeRecoveryFirmwareRequired(targetKey, out required);
            if (eStatus != ExitStatus.Success)
            {
                // MEMO: 復旧モード用FWの更新確認に失敗した場合は警告を出力だけ行って初期化は続行させる
                LOG.LogLine(LOG_LEVEL.LOG_WARN, "Recovery HostBridge FW check failed, but initialization will continue.");
                return ExitStatus.Success;
            }

            if (required)
            {
                eStatus = UpdateHostBridgeFirmware(targetKey, hostbridgeRecoveryFirmwarePath, true);
                if (eStatus != ExitStatus.Success)
                {
                    // MEMO: 復旧モード用FWの更新に失敗した場合は警告を出力だけ行って初期化は続行させる
                    LOG.LogLine(LOG_LEVEL.LOG_WARN, "Recovery HostBridge FW update failed, but initialization will continue.");
                    return ExitStatus.Success;
                }
            }
            return eStatus;
        }

        public ExitStatus IsUpdateHostBridgeRecoveryFirmwareRequired(TargetInnerClass targetKey, out bool required)
        {
            ExitStatus eStatus = ExitStatus.Failure;

            string targetIp = TargetSpecifier.GetTargetIp(targetKey);
            if (targetIp == null)
            {
                LOG.LogLine(LOG_LEVEL.LOG_WARN, "not found target ({0}).", targetKey.Value);
                required = false;
                return ExitStatus.Failure;
            }

            LOG.LogLine("Check HostBridge recovery FW update is required.");

            // Call UpdateHostBridge
            List<string> args1 = new List<string>() { "required", "--target", targetIp };
            List<string> argsList = new List<string>();
            argsList.AddRange(args1);

            ProcessAccessor.DumpProcessArgumentList("UpdateHostBridge", argsList);

            int retVar = Nintendo.UpdateHostBridge.CommandInterface.Main(argsList.ToArray());
            if(retVar == (int)Nintendo.UpdateHostBridge.CommandInterface.ExitCode.RecoveryUpdateRequired)
            {
                eStatus = ExitStatus.Success;
                required = true;
            }
            else if(retVar == (int)Nintendo.UpdateHostBridge.CommandInterface.ExitCode.RecoveryUpdateNotRequired)
            {
                eStatus = ExitStatus.Success;
                required = false;
            }
            else
            {
                eStatus = ExitStatus.Failure;
                required = false;
            }

            if (eStatus == ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_INFO2, "UpdateHostBridge passed.");
            }
            else
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "UpdateHostBridge failed.");
            }
            return eStatus;
        }

        private ExitStatus DoDevKitUpdate(string cardUpdaterPath, TargetInnerClass targetKey, string targetManagerPath)
        {
            LOG.LogLine("Do update.");

            string ipAddress = TargetSpecifier.GetTargetIp(targetKey);
            if (ipAddress == null)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Get IP address failed.");
                return ExitStatus.Failure;
            }

            // call RunOnTarget
            List<string> args1 = new List<string>() {"--target", ipAddress, "--reset",
                                "--failure-timeout", CardUpdateTimeoutSecond.ToString(),
                                "--pattern-not-found-failure", "Update Succeeded\\.", "--pattern-failure-exit", "Update Failed\\.",
                                "--extra-targetmanager-dir", targetManagerPath};
            List<string> args2 = new List<string>() { cardUpdaterPath };
            List<string> args3 = new List<string>() { "--", "--spmariobootpc" };

            if (Argument.Verbose)
            {
                args1.Add("--verbose");
            }
            List<string> argsList = new List<string>();
            argsList.AddRange(args1);
            argsList.AddRange(args2);

            ProcessAccessor.DumpProcessArgumentList("RunOnTarget", argsList);

            // not display
            argsList.AddRange(args3);

            int retVar = Nintendo.RunOnTarget.CommandInterface.Main(argsList.ToArray());

            // MEMO: put dummy line feed for RunOnTarget log
            Console.WriteLine(string.Empty);

#if false
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "  Wait 10 sec.");
            Thread.Sleep(10000);
#endif

            return retVar == 0 ? ExitStatus.Success : ExitStatus.Failure;
        }

        private ExitStatus RestartTarget(TargetInnerClass targetKey, string targetManagerPath)
        {
            // MEMO: This log is disabled because same log is outputted after.
            //LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Reboot target.");

            string targetIp = TargetSpecifier.GetTargetIp(targetKey);
            if (targetIp == null)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Get IP address failed.");
                return ExitStatus.Failure;
            }
            TargetManagerAccess.RebootTarget(targetIp, targetManagerPath, Argument.Verbose);

            Thread.Sleep(5000);

            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Reboot target finished.");

            return ExitStatus.Success;
        }

        private ExitStatus ActivateTarget(TargetInnerClass targetKey, string targetManagerPath)
        {
            return TargetManagerAccess.ActivateTarget(targetManagerPath, targetKey.Value) == InitializeSdev.ExitStatus.Success ?
                ExitStatus.Success : ExitStatus.Failure;
        }

        private bool FileCheck(string filePath)
        {
            if (!System.IO.File.Exists(filePath))
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "file not found : {0}", filePath);
                return false;
            }
            return true;
        }

        private void OutputDataReceived(object sender, DataReceivedEventArgs e)
        {
            Console.WriteLine(e.Data);
        }
    }
}
