﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading;
using Nintendo.InitializeSdev;
using static Nintendo.InitializeSdev.SdevInitializeCommandArgument;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;

namespace Nintendo.SystemUpdateSdev
{
    public enum ExitStatus
    {
        Success,
        Failure
    }

    public partial class SdevSystemUpdate
    {
        public SystemUpdateSdevCommandArgument Argument { get; private set; }
        private const int CardWriteTimeoutSecond = 90;
        private const int CardUpdateTimeoutSecond = 120;

        public SdevSystemUpdate()
        {
        }

        public ExitStatus Execute(SystemUpdateSdevCommandArgument arguments)
        {
            Argument = arguments;
            ExitStatus eStatus = ExitStatus.Failure;

            ProcessingProgress progress = ProcessingProgress.GetInstance();
            progress.SetProgress(0);

            FirmwareResourceSpecifier firmwareResource = new FirmwareResourceSpecifier();
            if (firmwareResource.SetVersion(GetVersionString()) == false)
            {
                return ExitStatus.Failure;
            }
            string targetManagerPath = firmwareResource.GetFileNameOfFirm(FirmwareType.Host_TargetManager);
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(targetManagerPath), targetManagerPath) == false)
            {
                return ExitStatus.Failure;
            }

#if false
            if (ExecutePluginsBeforeInitialize() != ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }
#endif

            progress.SetProgress(5);

            if (!arguments.keeptTargetManagerAlive)
            {
                LOG.LogLine("Stop NintendoTargetManager and check target information.");

                if (StopTargetManager() != ExitStatus.Success)
                {
                    return ExitStatus.Failure;
                }
            }

            progress.SetProgress(10);

            TargetInnerClass targetKey = new TargetInnerClass(string.Empty, TargetInnerClass.TargetType.TargetInner_Other);

            string targetArgumentString = null;
            TargetArgumentType taragetArgumentType = TargetArgumentType.TypeOther;
            if (!arguments.ParseTarget(ref targetArgumentString, ref taragetArgumentType))
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Cannot specify target. Please set --target.");
                return ExitStatus.Failure;
            }
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "SDEV target argument '{0}', type '{1}'", targetArgumentString, taragetArgumentType.ToString());

            TargetManagerAccess.UnregisterTarget(targetManagerPath, targetArgumentString, taragetArgumentType, Argument.Verbose);

            progress.SetProgress(15);

            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "initialize HostBridgeController first.");

            if (!TargetSpecifier.GetTargetKey(ref targetKey, targetArgumentString, taragetArgumentType))    // , arguments.connectIpDirect))
            {
                return ExitStatus.Failure;
            }
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "SDEV target key is {0} (type:{1})", targetKey.Value != null ? targetKey.Value : string.Empty, targetKey.Type.ToString());

            // RegisterTargetIfTargetTypeIsIp(targetKey, targetManagerPath);

            string hostbridgeFwPath = firmwareResource.GetFileNameOfFirm(FirmwareType.Firmware_HostBridge);
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(hostbridgeFwPath), hostbridgeFwPath) == false)
            {
                return ExitStatus.Failure;
            }

            progress.SetProgress(25);

            TargetSpecifier.SdevVersion sdevVersion;
#if true
            eStatus = CheckSdevVersion(out sdevVersion, targetKey);
            if (eStatus != ExitStatus.Success)
            {
                goto quit;
            }
#else
            sdevVersion = TargetSpecifier.SdevVersion.SDEV_1_8;
#endif
            firmwareResource.SetSdevVersion(sdevVersion);

            eStatus = ValidateSdevVersion(firmwareResource);
            if (eStatus != ExitStatus.Success)
            {
                goto quit;
            }

            // パスを取得する
            string devKitUpdaterPath = firmwareResource.GetFileNameOfFirm(FirmwareType.Firmware_DevKitUpdater);
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(devKitUpdaterPath), devKitUpdaterPath) == false)
            {
                return ExitStatus.Failure;
            }

            // Updaterを起動する
            eStatus = DoDevKitUpdate(devKitUpdaterPath, targetKey, targetManagerPath);
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "SystemUpdate failed.");
                goto quit;
            }

            progress.SetProgress(80);

#if false
            eStatus = RestartTarget(targetKey, targetManagerPath);
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Reboot target failed.");
                goto quit;
            }
#endif

            progress.SetProgress(90);

            eStatus = UpdateHostBridgeFirmware(targetKey, hostbridgeFwPath);
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Update HostBridge firmware failed.");
                goto quit;
            }

#if false
            string hostbridgeRecoveryFwPath = firmwareResource.GetFileNameOfFirm(FirmwareType.Firmware_HostBridgeRecovery);
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(hostbridgeRecoveryFwPath), hostbridgeRecoveryFwPath) == false)
            {
                goto quit;
            }

            eStatus = UpdateRecoveryHostBridgeFirmware(targetKey, hostbridgeRecoveryFwPath);
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Update Recovery HostBridge firmware failed.");
                goto quit;
            }
#endif

            eStatus = ActivateTarget(targetKey, targetManagerPath);
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_WARN, "Failed to activate target. ({0})", targetKey.Value);
            }

            if (!arguments.keeptTargetManagerAlive)
            {
                StopTargetManagerForLast();
            }

#if false
            if (ExecutePluginsAfterInitialize(targetKey, sdevVersion) != ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }
#endif

            LOG.LogLine("Sdev SystemUpdate SUCCEEDED.");

            progress.SetProgress(100);
            eStatus = ExitStatus.Success;
        quit:

            return eStatus;
        }
    }
}
