﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading;
using Nintendo.InitializeSdev;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;

namespace Nintendo.SystemUpdateEdev
{
    public enum ExitStatus
    {
        Success,
        Failure
    }

    partial class EdevSystemUpdate
    {
        public SystemUpdateEdevCommandArgument Argument { get; private set; }
        private const int DefaultTimeoutSecond = 600;
        private FirmwareResourceSpecifier firmwareResource;

        public ExitStatus Execute(SystemUpdateEdevCommandArgument arguments)
        {
            Argument = arguments;
            ExitStatus eStatus = ExitStatus.Failure;

            ProcessingProgress progress = ProcessingProgress.GetInstance();
            progress.SetProgress(0);

            if(UsbAccessCheck() != ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }

            progress.SetProgress(5);

            SdevInitializeCommandArgument dummySdevArguments = new SdevInitializeCommandArgument();
            firmwareResource = new FirmwareResourceSpecifier();
            if (firmwareResource.SetVersion(GetVersionString()) == false)
            {
                return ExitStatus.Failure;
            }
            string targetManagerPath = firmwareResource.GetFileNameOfFirm(FirmwareType.Host_TargetManager);
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(targetManagerPath), targetManagerPath) == false)
            {
                return ExitStatus.Failure;
            }

#if false
            if (ExecutePluginsBeforeInitialize() != ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }
#endif

            progress.SetProgress(10);

            LOG.LogLine("Stop NintendoTargetManager and check target information.");

            if (StopTargetManager() != ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }

            TargetInnerClass targetKey = new TargetInnerClass(string.Empty, TargetInnerClass.TargetType.TargetInner_Other);

            string targetArgumentString = null;
            InitializeEdev.EdevInitializeCommandArgument.TargetArgumentType taragetArgumentType = InitializeEdev.EdevInitializeCommandArgument.TargetArgumentType.TypeOther;
            if (!arguments.ParseTarget(ref targetArgumentString, ref taragetArgumentType))
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Cannot specify target. Please set --target.");
                return ExitStatus.Failure;
            }

            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "EDEV target argument '{0}', type '{1}'", targetArgumentString, taragetArgumentType.ToString());

            InitializeEdev.TargetManagerAccess.UnregisterTarget(targetManagerPath, targetArgumentString, taragetArgumentType, Argument.Verbose);

            progress.SetProgress(20);

            if (!InitializeEdev.TargetSpecifier.GetTargetKey(ref targetKey, targetArgumentString, taragetArgumentType, targetManagerPath))
            {
                return ExitStatus.Failure;
            }

            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "EDEV target key is {0} (type:{1})", targetKey.Value != null ? targetKey.Value : string.Empty, targetKey.Type.ToString());

            progress.SetProgress(25);

            progress.SetProgress(40);

            InitializeEdev.TargetSpecifier.EdevVersion edevVersion = InitializeEdev.TargetSpecifier.EdevVersion.EDEV_NONE;
            eStatus = GetEdevVersionInNormalMode(out edevVersion, targetKey, targetManagerPath, ref progress);
            if (eStatus != ExitStatus.Success)
            {
                goto quit;
            }
            Nintendo.InitializeSdev.TargetSpecifier.SdevVersion convertedSdevVerson = InitializeEdev.EdevInitializer.ConvertEdevToSdevVersion(edevVersion);
            firmwareResource.SetSdevVersion(convertedSdevVerson);

            eStatus = ValidateEdevVersion(firmwareResource);
            if (eStatus != ExitStatus.Success)
            {
                goto quit;
            }

            progress.SetProgress(55);

            // パスを取得する
            string devKitUpdaterPath = firmwareResource.GetFileNameOfFirm(FirmwareType.Firmware_DevKitUpdater);
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(devKitUpdaterPath), devKitUpdaterPath) == false)
            {
                return ExitStatus.Failure;
            }

            // Updaterを起動する
            // TODO: SDEV とは再起動方針が逆であることに注意
            eStatus = DoDevKitUpdate(devKitUpdaterPath, targetKey, targetManagerPath);
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "SystemUpdate failed.");
                goto quit;
            }

            // TODO: 再起動

#if false
            eStatus = DisconnectTarget(targetKey, targetManagerPath);
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_WARN, "Failed to disconnect target. ({0})", targetKey);
            }
#endif

            eStatus = ActivateTarget(targetKey, targetManagerPath);
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_WARN, "Failed to activate target. ({0})", targetKey);
            }

            StopTargetManagerForLast();

#if false
            if (ExecutePluginsAfterInitialize(targetKey, edevVersion) != ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }
#endif

            LOG.LogLine("Edev SystemUpdate SUCCEEDED.");

            progress.SetProgress(100);

            eStatus = ExitStatus.Success;

        quit:

            return eStatus;
        }
    }
}
