﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using System.Diagnostics;
using System.IO;
using System.Threading;
using System.Threading.Tasks;
using System.Threading.Tasks.Dataflow;
using Nintendo.ControlTarget;

namespace Nintendo.RunOnTarget
{
    public class WaitingUtility
    {
        public static ExitStatus WaitExitCondition(ISourceBlock<string> input, ExitCondition exitCondition, CancellationTokenSource cancelationSource, Task interruptTask)
        {
            ExitStatus exitStatus = ExitStatus.Failure;
            TimeSpan restSuccessTimeOut = exitCondition.SuccessTimeOut;
            TimeSpan restFailureTimeOut = exitCondition.FailureTimeOut;
            var isFoundNotFoundFailurePattern = (from pattern in exitCondition.NotFoundFailurePatterns select pattern).ToDictionary(v => v, v => false);
            var logExitCompletion = new TaskCompletionSource<bool>();

            if (exitCondition.NoWait)
            {
                Trace.WriteLine("Skip waiting exit.");
                return ExitStatus.Success;
            }

            Trace.WriteLine("Start waiting exit.");

            var monitorTask = Task.Factory.StartNew(() =>
            {
                try
                {
                    var halfwaySplitter = DataflowUtility.CreateHalfwayLineSplitter();
                    using (var unlink = DataflowUtility.LinkBlock("input->halfwaySplitter", input, halfwaySplitter))
                    {

                        while (true)
                        {
                            cancelationSource.Token.ThrowIfCancellationRequested();

                            var log = halfwaySplitter.Receive(cancelationSource.Token);

                            if (exitCondition.FailureExitPattern.IsMatch(log))
                            {
                                Trace.WriteLine($"Match the failure exit pattern. : {log}");
                                Console.WriteLine($"Match the failure exit pattern. : {log}");
                                exitStatus = ExitStatus.Failure;
                                return;
                            }

                            if (exitCondition.SuccessExitPattern.IsMatch(log))
                            {
                                Trace.WriteLine($"Match the success exit pattern. : {log}");
                                exitStatus = ExitStatus.Success;
                                return;
                            }

                            if (exitCondition.FailureTimeOutResetPattern.IsMatch(log))
                            {
                                Trace.WriteLine($"Match the failure timeout reset pattern. : {log}");
                                restFailureTimeOut = exitCondition.FailureTimeOut;
                            }

                            if (exitCondition.SuccessTimeOutResetPattern.IsMatch(log))
                            {
                                Trace.WriteLine($"Match the success timeout reset pattern. : {log}");
                                restSuccessTimeOut = exitCondition.SuccessTimeOut;
                            }

                            foreach (var pattern in exitCondition.NotFoundFailurePatterns)
                            {
                                if (pattern.IsMatch(log) && !isFoundNotFoundFailurePattern[pattern])
                                {
                                    Trace.WriteLine($"Match the not found failure pattern. : {log}");
                                    isFoundNotFoundFailurePattern[pattern] = true;
                                }
                            }

                            if (exitCondition.WaitExitProcess && log.Contains("\0\0\0\0"))
                            {
                                Trace.WriteLine("Match the exit log pattern.");
                                if (!logExitCompletion.Task.IsCompleted)
                                {
                                    logExitCompletion.SetResult(true);
                                    if (isFoundNotFoundFailurePattern.Values.All( v => v ))
                                    {
                                        exitStatus = ExitStatus.Success;
                                    }
                                    else
                                    {
                                        Console.WriteLine("!logExitCompletion.Task.IsCompleted");
                                        exitStatus = ExitStatus.Failure;
                                    }
                                }
                            }
                        }
                    }
                }
                catch (OperationCanceledException)
                {
                }
            }, cancelationSource.Token);

            var timeoutTask = Task.Factory.StartNew(async () =>
            {
                var interval = TimeSpan.FromSeconds(1);

                while (true)
                {
                    cancelationSource.Token.ThrowIfCancellationRequested();

                    if (restFailureTimeOut != TimeSpan.MaxValue)
                    {
                        if (restFailureTimeOut <= TimeSpan.Zero)
                        {
                            Trace.WriteLine("Failure timeout.");
                            Console.WriteLine("Failure timeout.");
                            exitStatus = ExitStatus.Failure;
                            return;
                        }

                        restFailureTimeOut -= interval;

                        Trace.WriteLine($"Rest time(failure timeout): {(int)restFailureTimeOut.TotalSeconds} sec");
                    }

                    if (restSuccessTimeOut != TimeSpan.MaxValue)
                    {
                        if (restSuccessTimeOut <= TimeSpan.Zero)
                        {
                            Trace.WriteLine("Success timeout.");
                            exitStatus = ExitStatus.Success;
                            return;
                        }

                        restSuccessTimeOut -= interval;

                        Trace.WriteLine($"Rest time(success timeout): {(int)restSuccessTimeOut.TotalSeconds} sec");
                    }

                    await Task.Delay(interval);
                }
            }, cancelationSource.Token).Unwrap();

            var waitProcessTask = Task.Factory.StartNew(async () =>
            {
                if (exitCondition.WaitExitProcess && exitCondition.TargetProcessCompletion != null)
                {
                    Trace.WriteLine("Wait exit process.");

                    exitCondition.TargetProcessCompletion.Wait(cancelationSource.Token);

                    if (exitCondition.TargetProcessCompletion.Result == 0)
                    {
                        logExitCompletion.Task.Wait(cancelationSource.Token);
                    }
                    else
                    {
                        Trace.WriteLine("Jit Debug Notification Received");
                        await Task.Delay(TimeSpan.FromSeconds(3));
                        cancelationSource.Cancel(true);
                        Console.WriteLine("Jit Debug Notification Received");
                        exitStatus = ExitStatus.Failure;
                        return;
                    }

                    // 暫定対処( http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-31559 )
                    // アプリの終了より前にシステムプロセスの終了マークが付くと
                    // not-found-failure-pattern が機能しないので、プログラムの終了時に判定を加える
                    // 本来はシステムプロセスのログの終了マークを何らかの手段で無視できるべき
                    if (logExitCompletion.Task.IsCompleted)
                    {
                        if (isFoundNotFoundFailurePattern.Values.All(v => v))
                        {
                            exitStatus = ExitStatus.Success;
                        }
                        else
                        {
                            Console.WriteLine("Not found failure pattern.");
                            exitStatus = ExitStatus.Failure;
                        }
                    }
                }
                else
                {
                    while (true)
                    {
                        var interval = TimeSpan.FromSeconds(2);
                        await Task.Delay(interval);

                        cancelationSource.Token.ThrowIfCancellationRequested();
                    }
                }
            }, cancelationSource.Token).Unwrap();

            if (interruptTask == null)
            {
                interruptTask = new Task(() => {
                    while (true)
                    {
                        Task.Delay(1000);
                    }
                }, cancelationSource.Token);
            }

            var exitTaskIndex = Task.WaitAny(monitorTask, timeoutTask, waitProcessTask, interruptTask);

            var taskNames = new string[] { "log monitor", "timeout", "process event", "interrupt" };
            Trace.WriteLine($"Exit waiting by {taskNames[exitTaskIndex]}");

            if (interruptTask.IsCompleted)
            {
                Trace.WriteLine("Interrupted by other process");
                Console.WriteLine("Interrupted by other process");
                exitStatus = ExitStatus.Failure;
            }

            cancelationSource.Cancel(true);

            WaitTaskAllowedCancel(monitorTask);
            monitorTask.Dispose();
            Trace.WriteLine("log monitor task exit.");

            WaitTaskAllowedCancel(timeoutTask);
            timeoutTask.Dispose();
            Trace.WriteLine("timeout task exit.");

            WaitTaskAllowedCancel(waitProcessTask);
            waitProcessTask.Dispose();
            Trace.WriteLine("waiting process task exit.");

            WaitTaskAllowedCancel(interruptTask);
            interruptTask.Dispose();
            Trace.WriteLine("waiting interrupt task exit.");

            Trace.WriteLine($"Exit program. Status = {exitStatus}");
            if (exitStatus == ExitStatus.Failure)
            {
                Console.WriteLine($"WaitExitCondition() : {exitStatus}");
            }

            return exitStatus;
        }

        public static void WaitTaskAllowedCancel(Task task)
        {
            var tokenSource = new CancellationTokenSource();
            WaitTaskAllowedCancel(task, tokenSource.Token);
        }

        public static void WaitTaskAllowedCancel(Task task, CancellationToken token)
        {
            try
            {
                task.Wait(token);
            }
            catch (OperationCanceledException)
            {
            }
            catch (AggregateException e)
            {
                foreach (var exception in e.InnerExceptions)
                {
                    try
                    {
                        throw exception;
                    }
                    catch (OperationCanceledException)
                    {
                    }
                }
            }
        }
    }
}
