﻿namespace Nintendo.NintendoSdkServiceManager
{
    using System;
    using System.IO;
    using System.Diagnostics;
    using Nintendo.Foundation.IO;
    using System.Collections.Generic;
    using NintendoSdkServiceManager;

    internal class Program
    {
        internal static int Main(string[] args)
        {
            try
            {
                new ConsoleApplication().Run(args);
                return 0;
            }
            catch
            {
                return 1;
            }
        }
    }

    /// <summary>
    /// ServiceManager アプリケーションクラスです。
    /// </summary>
    internal class ConsoleApplication
    {
        /// <summary>
        /// アプリケーションを実行します。
        /// </summary>
        /// <param name="args">コマンドライン引数を指定します。</param>
        /// <returns>成功したら true、失敗したら false を返します。</returns>
        public void Run(string[] args)
        {
            try
            {
                var parser = new Nintendo.Foundation.IO.CommandLineParser();
                ServiceManagerArgument parsed;
                if (false == parser.ParseArgs<ServiceManagerArgument>(args, out parsed))
                {
                    string message = "[" + NowString + "] [ERROR       ] Cannot parse command line args.";
                    throw new Exception(message);
                }
                else if (parsed.Start != null)
                {
                    Start();
                }
                else if (parsed.Stop != null)
                {
                    Stop();
                }
                else if (parsed.List != null)
                {
                    List();
                }
                else
                {
                    string message = "[" + NowString + "] [ERROR       ] Unknown command line args.";
                    throw new Exception(message);
                }
            }
            catch (Exception exception)
            {
                using (var file = new System.IO.StreamWriter(LogFile, true, System.Text.Encoding.UTF8))
                {
                    file.WriteLine("[{0}] [ERROR       ] {1}", NowString, exception.Message);
                    file.WriteLine(exception.StackTrace);
                }
                throw;
            }
        }

        /// --------------------------------------------------------------------
        /// サブコマンド処理
        /// --------------------------------------------------------------------

        private void Start()
        {
            var startRequestInfoList = GetStartRequestInfoList();

            // ↑で得たコマンドを１つずつ実行していく。
            //途中で失敗したら、その場で false を返して Start を抜ける。すべてうまくいけば true を返す。
            // 成功したコマンドは StartedProcessesFile に書いていく。
            StartProcess(startRequestInfoList);
        }

        private void Stop()
        {
            // ① Start の ③ で書いた StartedProcessesFile を開き、実行したコマンドのリストを得る。なければ何もしない。
            var startedProcessInfoList = GetStartedProcessInfoList();

            // ② ① で得たコマンドを１つずつ殺していく。当該コマンドが実行中でない場合は何もしない。
            //    同じコマンドが複数起動していたらすべて殺す。すべて殺し終わったら true を返す。
            StopProcess(startedProcessInfoList);
        }

        private void List()
        {
            var startRequestInfoList = GetStartRequestInfoList();

            // ↑ で得たリストをフルパスで Console.Out で出力。
            ListProcess(startRequestInfoList);
        }

        /// --------------------------------------------------------------------
        /// 各種実装処理
        /// --------------------------------------------------------------------
        private bool IsValidStartRequestInfoLine(string line)
        {
            if (line == string.Empty)
            {
                // 空行なのでスキップ
                return false;
            }
            if (line[0] == '#')
            {
                // # はじまりなのでスキップ
                return false;
            }
            return true;
        }

        private List<StartRequestInfo> GetStartRequestInfoList()
        {
            var sdkDir = NintendoSdkServiceManager.NintendoSdkRootMarkFinder.Find();

            // sdkDir の下に ServiceProcessList.txt があるか？を確認
            var processListFile = Path.Combine(sdkDir, @"Tools\CommandLineTools\ServiceProcessList.txt");
            if (false == File.Exists(processListFile))
            {
                throw new FileNotFoundException(processListFile + " file is not found.");
            }

            // 「スタートしたいプロセスイメージのパスと、引数」のリストを構築
            var processListFileDir = Path.Combine(sdkDir, @"Tools\CommandLineTools");
            var infoList = new List<StartRequestInfo>();
            foreach (var line in File.ReadAllLines(processListFile))
            {
                if (IsValidStartRequestInfoLine(line))
                {
                    var tmp = line.Split(new char[] {' '}, 2);

                    var info = new StartRequestInfo();
                    info.ProcessPath = Path.GetFullPath(Path.Combine(processListFileDir, tmp[0]));
                    info.Args = (tmp.Length) >= 2 ? tmp[1] : string.Empty;
                    infoList.Add(info);
                }
            }

            return infoList;
        }

        private void StartProcess(List<StartRequestInfo> infoList)
        {
            using (System.IO.StreamWriter logFile = new System.IO.StreamWriter(LogFile, true, System.Text.Encoding.UTF8),
                   startedProcessesFile = new System.IO.StreamWriter(StartedProcessesFile, false, System.Text.Encoding.UTF8))
            {
                foreach (var info in infoList)
                {
                    if (false == File.Exists(info.ProcessPath))
                    {
                        throw new FileNotFoundException(info.ProcessPath + " file is not found.");
                    }

                    // プロセス実行
                    var process = System.Diagnostics.Process.Start(info.ProcessPath, info.Args);

                    // ログ出力
                    // MEMO: パス文字列としては利用できない "|" をデリミタとして利用。
                    //       1234|2016/04/06 14:29:54|C:\hoge\piyo\fuga.exe のような行ができる。
                    var logMessage = process.Id + "|" + process.StartTime.ToString() + "|" + process.MainModule.FileName;
                    startedProcessesFile.WriteLine(logMessage);
                    logFile.WriteLine("[{0}] [START       ] {1}", NowString, logMessage);
                }
            }
        }

        private void StopProcess(ProcessInfo[] startedProcessInfoList)
        {
            using (var file = new System.IO.StreamWriter(LogFile, true, System.Text.Encoding.UTF8))
            {
                foreach (var info in startedProcessInfoList)
                {
                    string dumpMessageFragment = info.Id + "|" + info.StartTime + "|" + info.ProcessPath;

                    var p = SearchTargetProcess(info);
                    if (p != null)
                    {
                        if (p.MainWindowHandle != IntPtr.Zero)
                        {
                            if (false == p.CloseMainWindow())
                            {
                                file.WriteLine("[{0}] [CANNOT CLOSE] {1}", NowString, dumpMessageFragment);
                                string message = "Cannot close: " + dumpMessageFragment;
                                throw new Exception(message);
                            }
                        }
                        else
                        {
                            TaskTrayApplicationTerminator.Run(info.Id, info.ProcessPath);
                        }

                        if (p.WaitForExit(StopTimeOutMilliseconds))
                        {
                            file.WriteLine("[{0}] [CLOSE       ] {1}", NowString, dumpMessageFragment);
                        }
                        else
                        {
                            p.Kill();
                            file.WriteLine("[{0}] [FORCE CLOSE ] {1}", NowString, dumpMessageFragment);
                        }
                    }
                    else
                    {
                        file.WriteLine("[{0}] [NOT FOUND   ] {1}", NowString, dumpMessageFragment);
                    }
                }
            }
        }

        private void ListProcess(List<StartRequestInfo> infoList)
        {
            foreach (var info in infoList)
            {
                var resultString = info.ProcessPath;
                if (false == File.Exists(info.ProcessPath))
                {
                    resultString += " [NOT FOUND]";
                }
                resultString += " (" + info.Args + ")";
                Console.WriteLine(resultString);
            }
        }

        private class StartRequestInfo
        {
            public string ProcessPath { get; set; } /// 絶対パスで格納する
            public string Args { get; set; }
        }

        private class ProcessInfo
        {
            public int Id { get; set; }
            public string StartTime { get; set; }
            public string ProcessPath { get; set; }
        }

        private ProcessInfo[] GetStartedProcessInfoList()
        {
            if (false == File.Exists(StartedProcessesFile))
            {
                return null;
            }

            string[] logInfoList = File.ReadAllLines(StartedProcessesFile);
            if (logInfoList.Length < 1)
            {
                return null;
            }

            ProcessInfo[] processInfoList = new ProcessInfo[logInfoList.Length];
            for (var i = 0; i < logInfoList.Length; i++)
            {
                var info = new ProcessInfo();
                processInfoList[i] = info;

                var tmp = logInfoList[i].Split('|');
                info.Id = int.Parse(tmp[0]);
                info.StartTime = tmp[1];
                info.ProcessPath = tmp[2];
            }
            return processInfoList;
        }

        private Process SearchTargetProcess(ProcessInfo target)
        {
            foreach (var p in System.Diagnostics.Process.GetProcesses())
            {
                if (p.Id == target.Id &&
                    p.StartTime.ToString() == target.StartTime &&
                    p.MainModule.FileName == target.ProcessPath)
                {
                    return p;
                }
            }
            return null;
        }

        /// --------------------------------------------------------------------
        /// 定数
        /// --------------------------------------------------------------------
        private static readonly string NowString            = DateTime.Now.ToString();
        private static readonly string AppDataDir           = Path.Combine(Environment.GetEnvironmentVariable("APPDATA"), @"Nintendo\Oasis\TargetManager");
        private static readonly string LogFile              = Path.Combine(AppDataDir, @"NintendoSdkServiceManager.Log.txt");
        private static readonly string StartedProcessesFile = Path.Combine(AppDataDir, @"NintendoSdkServiceManager.StartedProcesses.txt");
        private static readonly int StopTimeOutMilliseconds = 10000;

        /// --------------------------------------------------------------------
        /// コマンドライン引数の設定
        /// --------------------------------------------------------------------
        /// <summary>
        /// コマンドライン引数の設定
        /// </summary>
        private class ServiceManagerArgument
        {
            [CommandLineSubCommand("start", Description = "start NintendoSDK service processes.")]
            public StartCommand Start { get; set; }

            [CommandLineSubCommand("stop", Description = "stop NintendoSDK service processes.")]
            public StopCommand Stop { get; set; }

            // おもにデバッグ用
            [CommandLineSubCommand("list", Description = "dump NintendoSDK path & list NintendoSDK service processes.")]
            public ListCommand List { get; set; }
        }

        private class StartCommand
        {
        }

        private class StopCommand
        {
        }

        private class ListCommand
        {
        }
    }
}
