﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Diagnostics;
using System.Threading;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;
using Nintendo.Bridge;

namespace Nintendo.InitializeSdev
{
    public class TargetSpecifier
    {
#if false
        private HostBridgeController m_Handle;

        private TargetSpecifier()
        {
            m_Handle = null;
        }

        public TargetSpecifier(HostBridgeController handle)
        {
            m_Handle = handle;
        }

        public void SetHandle(HostBridgeController handle)
        {
            m_Handle = handle;
        }

        public void Release()
        {
            m_Handle.Release();
        }
#endif

        public static bool GetTargetKey(ref TargetInnerClass targetKey, string targetArgumentString, SdevInitializeCommandArgument.TargetArgumentType targetArgumentType, bool IsConnectDirectIp = false)
        {
            if (IsConnectDirectIp)
            {
                if(targetArgumentType == SdevInitializeCommandArgument.TargetArgumentType.TypeIP)
                {
                    targetKey.Set(targetArgumentString, TargetInnerClass.TargetType.TargetInner_SdevIp_Force);
                    return true;
                }
                else
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "The target string is not IP address ({0}), despite specifying --connect-ip-direct.", targetArgumentString);
                    return false;
                }
            }

            switch (targetArgumentType)
            {
                case SdevInitializeCommandArgument.TargetArgumentType.TypeIP:
                    targetKey.Set(GetTargetMacFromIp(targetArgumentString), TargetInnerClass.TargetType.TargetInner_SdevMac);
                    if (targetKey.Value == null)
                    {
                        LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Failed to connect target by IP ({0}).", targetArgumentString);
                    }
                    else
                    {
                        return true;
                    }
                    break;
                case SdevInitializeCommandArgument.TargetArgumentType.TypeName:
                    targetKey.Set(GetTargetMacFromName(targetArgumentString), TargetInnerClass.TargetType.TargetInner_SdevMac);
                    if (targetKey.Value == null)
                    {
                        LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Failed to connect target by name ({0}).", targetArgumentString);
                    }
                    else
                    {
                        return true;
                    }
                    break;
                case SdevInitializeCommandArgument.TargetArgumentType.TypeSerial:
                    SerialNumber serialNumber = new SerialNumber(targetArgumentString);
                    if (serialNumber.GetString() == null)
                    {
                        LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Invalid serial number ({0}).", targetArgumentString);
                        break;
                    }
                    targetKey.Set(GetTargetMacFromSerial(targetArgumentString), TargetInnerClass.TargetType.TargetInner_SdevMac);
                    if (targetKey.Value == null)
                    {
                        LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Failed to connect target by serial number ({0}).", targetArgumentString);
                    }
                    else
                    {
                        return true;
                    }
                    break;
                case SdevInitializeCommandArgument.TargetArgumentType.TypeMacAddress:
                    targetKey.Set(targetArgumentString, TargetInnerClass.TargetType.TargetInner_SdevMac);
                    return true;
            }

            return false;
        }

        private static string ConvertTMMacToMac(string targetMacForTM)
        {
            if (targetMacForTM != null)
            {
                return targetMacForTM.Replace('-', ':').ToUpper();
            }
            return null;
        }

        private static string ConvertMacToTMMac(string targetMac)
        {
            if (targetMac != null)
            {
                return targetMac.Replace(':', '-').ToLower();
            }
            return null;
        }

        public static string GetTargetMacFromIp(string targetIp)
        {
            string targetMacForTM = ControlTarget.Utility.GetTargetMacFromIp(targetIp);
            return ConvertTMMacToMac(targetMacForTM);
        }

        public static string GetTargetMacFromName(string targetName)
        {
            string targetMacForTM = ControlTarget.Utility.GetTargetMacFromName(targetName);
            return ConvertTMMacToMac(targetMacForTM);
        }

        public static string GetTargetMacFromSerial(string targetSerial)
        {
            string targetMacForTM = ControlTarget.Utility.GetTargetMacFromSerial(targetSerial);
            return ConvertTMMacToMac(targetMacForTM);
        }

        public static string GetTargetIpFromMac(string targetMac)
        {
            string targetMacForTM = ConvertMacToTMMac(targetMac);
            return ControlTarget.Utility.GetTargetIpFromMac(targetMacForTM);
        }

        public static string GetTargetSerialFromMac(string targetMac)
        {
            string targetMacForTM = ConvertMacToTMMac(targetMac);
            return ControlTarget.Utility.GetTargetSerialFromMac(targetMacForTM);
        }

        public static string GetTargetSerialFromIp(string targetIp)
        {
            return ControlTarget.Utility.GetTargetSerialFromIp(targetIp);
        }

        public static string GetTargetIp(TargetInnerClass target)
        {
            switch (target.Type)
            {
                case TargetInnerClass.TargetType.TargetInner_SdevMac:
                    return GetTargetIpFromMac(target.Value);
                case TargetInnerClass.TargetType.TargetInner_SdevIp_Force:
                    return target.Value;
                default:
                    return null;
            }

        }

        public static ExitStatus EnumerateTargets(out string[] ips, out int[] ports, out string[] names, out string[] macs, out string[] serials)
        {
            BridgeDetect detector = new Bridge.BridgeDetect();

            int targetNum = 0;
            UInt32[] versions = null;
            detector.Enumerate(out targetNum, out ips, out ports, out names, out macs, out versions, out serials, 3000);
            return ExitStatus.Success;
        }

        public enum SdevVersion
        {
            SDEV_1_5,
            SDEV_1_6,
            SDEV_1_6_G,
            SDEV_1_8,
            SDEV_PRE_MP1,
            SDEV_PRE_MP2,
            SDEV_MP,
            EDEV_EP_2_1,
            EDEV_EP_2_2,
            EDEV_MP,
            SDEV_NONE,
            SDEV_ERROR
        };

        public static bool IsEdev(SdevVersion devkitVersion)
        {
            switch(devkitVersion)
            {
                case SdevVersion.EDEV_EP_2_1:
                case SdevVersion.EDEV_EP_2_2:
                case SdevVersion.EDEV_MP:
                    return true;
                default:
                    return false;
            }
        }

        public static bool IsSdev(SdevVersion devkitVersion)
        {
            switch (devkitVersion)
            {
                case SdevVersion.SDEV_1_5:
                case SdevVersion.SDEV_1_6:
                case SdevVersion.SDEV_1_6_G:
                case SdevVersion.SDEV_1_8:
                case SdevVersion.SDEV_PRE_MP1:
                case SdevVersion.SDEV_PRE_MP2:
                case SdevVersion.SDEV_MP:
                    return true;
                default:
                    return false;
            }
        }

        public static SdevVersion GetSdevVersion(TargetInnerClass targetKey, bool isVerbose = false)
        {
            string targetIp = GetTargetIp(targetKey);
            if (targetIp == null)
            {
                LOG.LogLine(LOG_LEVEL.LOG_WARN, "not found target ({0}).", targetKey.Value);
                return SdevVersion.SDEV_ERROR;
            }

            BridgeInfo info = new BridgeInfo();

            string hwVersion = null;
            bool result = false;
            const int MaxRetryCount = 32;

            for (int cnt = 0; cnt < MaxRetryCount; cnt++)
            {
                result = info.GetHwVersion(targetIp, out hwVersion);

                if (result == false)
                {
                    LOG.LogLine(LOG_LEVEL.LOG_WARN, "Get SDEV version failed. Retry ...");
                    Thread.Sleep(TimeSpan.FromSeconds(1));
                    continue;
                }
            }

            if (result == false)
            {
                return SdevVersion.SDEV_ERROR;
            }

            switch (hwVersion)
            {
                case "1":
                    return SdevVersion.SDEV_1_5;
                case "2":
                    return SdevVersion.SDEV_1_6;
                case "3":
                    return SdevVersion.SDEV_1_8;
                case "4":
                    return SdevVersion.SDEV_1_6_G;
                case "5":
                    return SdevVersion.SDEV_PRE_MP1;
                case "6":
                    return SdevVersion.SDEV_PRE_MP2;
                case "7":
                    return SdevVersion.SDEV_MP;
                default:
                    LOG.LogLine(LOG_LEVEL.LOG_INFO2, "config.HwVersion = {0}", hwVersion);
                    return SdevVersion.SDEV_NONE;
            }
        }
    }

    public class TargetInnerClass
    {
        public enum TargetType
        {
            TargetInner_SdevMac,
            TargetInner_SdevIp_Force,
            TargetInner_EdevSerial,
            TargetInner_EdevConnectedUsb,
            TargetInner_Other
        }

        private string m_String;
        private TargetType m_Type;

        private TargetInnerClass() { }

        public TargetInnerClass(string targetString, TargetType type)
        {
            Set(targetString, type);
        }

        public string Value
        {
            get { return m_String; }
        }

        public TargetType Type
        {
            get { return m_Type; }
        }

        public void Set(string targetString, TargetType type)
        {
            m_String = targetString;
            m_Type = type;
        }
    }
}
