﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Text.RegularExpressions;
using Nintendo.Foundation.IO;
using Nintendo.ControlTarget;

namespace Nintendo.InitializeSdev
{
    public class SdevInitializeCommandArgument
    {
        [CommandLineOption("target",
            Description = "Specifies IP, serial number or name of target SDEV.")]
        public string target { get; set; }

        [CommandLineOption("target-ip", IsHidden = true,
            Description = "[for debug] Specifies IP address of target SDEV.")]
        public string targetIp { get; set; }

        [CommandLineOption("target-name", IsHidden = true,
            Description = "[for debug] Specifies name of target SDEV.")]
        public string targetName { get; set; }

        [CommandLineOption("target-serial", IsHidden = true,
            Description = "[for debug] Specifies name of target SDEV.")]
        public string targetSerial { get; set; }

        [CommandLineOption("target-mac", IsHidden = true,
            Description = "[for debug] Specifies MAC address of target SDEV.")]
        public string targetMac { get; set; }

        [CommandLineOption("connect-ip-direct",
            Description = "Connect to SDEV by IP directly. This option is for connecting to other segment.")]
        public bool connectIpDirect { get; set; }

        [CommandLineOption("target-version",
            Description = "Specifies firmware variation.")]
        public string targetFirmwareVersion { get; set; }

        [CommandLineOption("display-available-version",
            Description = "Display available firmware variation.")]
        public bool displayAvailableVersion { get; set; }

        [CommandLineOption("disable-hostbridge-fw-update", IsHidden = true,
            Description = "[for debug] Disable HostBridge firmware update.")]
        public bool disableHostBridgeFirmwareUpdate { get; set; }

        [CommandLineOption("list-available-targets", IsHidden = true,
            Description = "List available targets.")]
        public bool listAvailableTargets { get; set; }

        [CommandLineOption("verbose",
            Description = "Log verbosely.")]
        public bool Verbose { get; set; }

        [CommandLineOption("ndi-mode", IsHidden = true,
            Description = "[for ndi] Change output mode for ndi.")]
        public string ndiPipeName { get; set; }

        [CommandLineOption("ndi-locale", IsHidden = true,
            Description = "[for ndi] Specify output locale.")]
        public string ndiLocale { get; set; }

        [CommandLineOption("keep-targetmanager-alive",
            Description = "Do not close TargetManager for concurrent initialization.")]
        public bool keeptTargetManagerAlive { get; set; }

        public enum TargetArgumentType
        {
            TypeIP,
            TypeName,
            TypeSerial,
            TypeMacAddress,
            TypeOther
        }

        private bool CheckIpAddressString(string targetString)
        {
            return Regex.IsMatch(targetString, @"^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$");
        }

        private bool CheckMacAddressString(string targetString)
        {
            // XX:XX:XX:XX:XX:XX 形式のみサポート
            return Regex.IsMatch(targetString, @"^([0-9A-Fa-f]{2}[:]){5}[0-9A-Fa-f]{2}$");

        }

        public bool ParseTarget(ref string targetString, ref TargetArgumentType targetType)
        {
            if (this.targetIp != null)
            {
                targetString = this.targetIp;
                targetType = TargetArgumentType.TypeIP;
                return true;
            }
            else if (this.targetName != null)
            {
                targetString = this.targetName;
                targetType = TargetArgumentType.TypeName;
                return true;
            }
            else if (this.targetSerial != null)
            {
                targetString = this.targetSerial;
                targetType = TargetArgumentType.TypeSerial;
                return true;
            }
            else if (this.targetMac != null)
            {
                targetString = this.targetMac;
                targetType = TargetArgumentType.TypeMacAddress;
                return true;
            }
            else if (this.target != null)
            {
                if (CheckIpAddressString(this.target))
                {
                    targetString = this.target;
                    targetType = TargetArgumentType.TypeIP;
                }
                else if (CheckMacAddressString(this.target))
                {
                    targetString = this.target;
                    targetType = TargetArgumentType.TypeMacAddress;
                }
                else if (SerialNumber.CheckSerialString(this.target))
                {
                    targetString = this.target;
                    targetType = TargetArgumentType.TypeSerial;
                }
                else
                {
                    targetString = this.target;
                    targetType = TargetArgumentType.TypeName;
                }
                return true;
            }
            return false;
        }
    }
}
