﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading;
using System.IO;
using System.Diagnostics;
using Nintendo.InitializeSdev;
using System.Text.RegularExpressions;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;

namespace Nintendo.InitializeEdev
{
    // MEMO: セーフモード処理のうち旧セーフモードのものについては SafeModeOld.cs に記述してあります

    partial class EdevInitializer
    {
        private string ProcessOutputBuffer = null;

        private ExitStatus CheckNewSafeMode(out bool isNewSafeMode, TargetInnerClass targetKey, string targetManagerPath, out bool isSerialIdFound)
        {
            ExitStatus eStatus = ExitStatus.Failure;
            isNewSafeMode = false;
            isSerialIdFound = true;

            SerialNumber serial = new SerialNumber(targetKey.Value);
            if (serial.GetString() == null)
            {
                return ExitStatus.Failure;
            }
            string targetStringWithDigit = serial.GetStringWithCheckDigit();

            string recoveryWriterUsbPath = firmwareResource.GetFileNameOfFirm(FirmwareType.Host_RecoveryWriterUsb);
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(recoveryWriterUsbPath), recoveryWriterUsbPath) == false)
            {
                return ExitStatus.Failure;
            }

            // RecoveryWriterUsb で表示されていれば true
            eStatus = SafeMode.GetSafeModeSerialNumbers(recoveryWriterUsbPath, Argument.Verbose);
            if (eStatus != ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }
            if (!SafeMode.IsExistSafeModeSerialNumberList(targetStringWithDigit))
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Serial ID is not found. ({0})", targetStringWithDigit);
                isSerialIdFound = false;
                return ExitStatus.Success;
            }

            eStatus = SafeMode.GetBatteryCharge(recoveryWriterUsbPath, targetStringWithDigit, Argument.Verbose);
            if (eStatus == ExitStatus.Success)
            {
                isNewSafeMode = true;
                return ExitStatus.Success;
            }

            // 以下旧セーフモードであると想定したコード

            return ExitStatus.Success;
        }

        private ExitStatus ExecuteInSafeMode(out TargetSpecifier.EdevVersion edevVersion, TargetInnerClass targetKey, ref ProcessingProgress progress)
        {
            ExitStatus eStatus = ExitStatus.Failure;
            // MEMO: この関数全体での progress 増分は 35 の想定
            int safemodeStartProgress = progress.GetProgress();

            string recoveryWriterUsbPath = firmwareResource.GetFileNameOfFirm(FirmwareType.Host_RecoveryWriterUsb);
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(recoveryWriterUsbPath), recoveryWriterUsbPath) == false)
            {
                edevVersion = TargetSpecifier.EdevVersion.EDEV_ERROR;
                return ExitStatus.Failure;
            }

            SerialNumber serial = new SerialNumber(targetKey.Value);
            if (serial.GetString() == null)
            {
                edevVersion = TargetSpecifier.EdevVersion.EDEV_ERROR;
                return eStatus;
            }
            string targetStringWithDigit = serial.GetStringWithCheckDigit();

            // TODO: modify value
            progress.SetProgress(safemodeStartProgress + 10);

#if true
            eStatus = SafeMode.CheckEdevVersion(out edevVersion, targetStringWithDigit, recoveryWriterUsbPath, Argument.Verbose);
            if (eStatus != ExitStatus.Success)
            {
                goto quit;
            }
#else
            edevVersion = TargetSpecifier.EdevVersion.EDEV_EP_2_2;
#endif
            Nintendo.InitializeSdev.TargetSpecifier.SdevVersion convertedSdevVerson = ConvertEdevToSdevVersion(edevVersion);
            firmwareResource.SetSdevVersion(convertedSdevVerson);

            progress.SetProgress(safemodeStartProgress + 15);

            string qspiBootImagePath = firmwareResource.GetFileNameOfFirm(FirmwareType.Firmware_QspiBootImage);
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(qspiBootImagePath), qspiBootImagePath) == false)
            {
                return ExitStatus.Failure;
            }

            eStatus = SafeMode.UpdateRecoveryBootImage(targetStringWithDigit, qspiBootImagePath, recoveryWriterUsbPath, Argument.Verbose);
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Update Recovery boot image failed.");
                goto quit;
            }

            eStatus = ExitStatus.Success;

        quit:
            return eStatus;
        }
    }

    public static class SafeMode
    {
        private const int BatteryLowValue = 5;      // バッテリーの下限
        private static string ProcessOutputBuffer = null;
        private static bool IsVerbose = false;
        private const string RecoveryWriterUsbSuccessString = "[SUCCESS]";
        private const string ConvertedRecoveryWriterUsbSuccessString = "[INFO](log)";

        private static int InSafeModeTargetNum;
        private static List<string> SafeModeSerialNumberList;
        private static int BatteryBalue;

        private static Dictionary<TargetSpecifier.EdevVersion, string> edevVersionDictionary = new Dictionary<TargetSpecifier.EdevVersion, string>()
        {
            {TargetSpecifier.EdevVersion.EDEV_EP_2_1, "EP2-1"},
            {TargetSpecifier.EdevVersion.EDEV_EP_2_2, "EP2-2"},
            {TargetSpecifier.EdevVersion.EDEV_MP, "MP"},
            {TargetSpecifier.EdevVersion.EDEV_NONE, ""},
            {TargetSpecifier.EdevVersion.EDEV_ERROR, ""},
        };

        public static ExitStatus GetSafeModeSerialNumbers(string recoveryWriterUsbPath, bool isVerbose = false)
        {
            ExitStatus eStatus = ExitStatus.Failure;

            InSafeModeTargetNum = 0;
            SafeModeSerialNumberList = new List<string>();

            // call RecoveryWriterUsb
            List<string> argsList = new List<string>();
            List<string> args1 = new List<string>() { "GetSerialNumber" };
            argsList.AddRange(args1);

            ProcessAccessor.DumpProcessArgumentList("RecoveryWriterUsb", argsList);

            ProcessOutputBuffer = null;
            IsVerbose = isVerbose;
            int exitCode = ProcessAccessor.DoProcess(recoveryWriterUsbPath, argsList, RecoveryWriterUsbOutputDataReceived);

            if (exitCode != 0 || !CheckRecoveryWriterUsbSucceeded())
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "GetSerialNumber command failed.");
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, ConstData.GetEdevErrorInfomationString());
                goto quit;
            }

            if (!ParseRecoveryWriterUsbSerialNumberOutput())
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Parse GetSerialNumber output failed.");
                goto quit;
            }

            eStatus = ExitStatus.Success;

        quit:
            return eStatus;
        }

        public static ExitStatus GetBatteryCharge(string recoveryWriterUsbPath, string serialNumberWithDigit, bool isVerbose = false)
        {
            ExitStatus eStatus = ExitStatus.Failure;

            InSafeModeTargetNum = 0;
            SafeModeSerialNumberList = new List<string>();

            // call RecoveryWriterUsb
            List<string> argsList = new List<string>();
            List<string> args1 = new List<string>() { "GetBatteryCharge", serialNumberWithDigit };
            argsList.AddRange(args1);

            ProcessAccessor.DumpProcessArgumentList("RecoveryWriterUsb", argsList);

            ProcessOutputBuffer = null;
            IsVerbose = isVerbose;
            int exitCode = ProcessAccessor.DoProcess(recoveryWriterUsbPath, argsList, RecoveryWriterUsbOutputDataReceived);

            if (exitCode != 0 || !CheckRecoveryWriterUsbSucceeded())
            {
                // 旧セーフモードの場合もこのシーケンスに入るためエラーではなく警告にする
                LOG.LogLine(LOG_LEVEL.LOG_WARN, "GetBatteryCharge command failed.");
                goto quit;
            }

            if (!ParseRecoveryWriterUsbBatteryChargeOutput())
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Parse GetBatteryCharge output failed.");
                goto quit;
            }

            // バッテリー残量チェック
            if (BatteryBalue < BatteryLowValue)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Battery is LOW ! Please charge the EDEV.");
                goto quit;
            }

            eStatus = ExitStatus.Success;

        quit:
            return eStatus;
        }

        public static ExitStatus CheckEdevVersion(out TargetSpecifier.EdevVersion edevVersion, string targetStringWithDigit, string recoveryWriterUsbPath, bool isVerbose = false)
        {
            ExitStatus eStatus = ExitStatus.Failure;
            edevVersion = TargetSpecifier.EdevVersion.EDEV_ERROR;

            // call RecoveryWriterUsb
            List<string> argsList = new List<string>();
            List<string> args1 = new List<string>() { "GetConfigurationId1", targetStringWithDigit };
            argsList.AddRange(args1);

            ProcessAccessor.DumpProcessArgumentList("RecoveryWriterUsb", argsList);

            ProcessOutputBuffer = null;
            IsVerbose = isVerbose;
            int exitCode = ProcessAccessor.DoProcess(recoveryWriterUsbPath, argsList, RecoveryWriterUsbOutputDataReceived);

            if (exitCode != 0 || !CheckRecoveryWriterUsbSucceeded())
            {
                goto quit;
            }
            if (!ParseConfigurationId1Output(out edevVersion))
            {
                goto quit;
            }

            eStatus = ExitStatus.Success;

        quit:
            return eStatus;
        }

        public static ExitStatus UpdateRecoveryBootImage(string targetStringWithDigit, string qspiBootImagePath, string recoveryWriterUsbPath, bool isVerbose = false)
        {
            LOG.LogLine("Update Recovery boot image.");

            ExitStatus eStatus = ExitStatus.Failure;

            // call RecoveryWriterUsb

            string qspiBootImageDirectory = Path.GetDirectoryName(qspiBootImagePath);
            string qspiBootImageFile = Path.GetFileName(qspiBootImagePath);
            string currentDirectory = Directory.GetCurrentDirectory();

            Directory.SetCurrentDirectory(qspiBootImageDirectory);

            List<string> argsList = new List<string>();
            List<string> args1 = new List<string>() { "SendImage", targetStringWithDigit, qspiBootImageFile };
            argsList.AddRange(args1);

            ProcessAccessor.DumpProcessArgumentList("RecoveryWriterUsb", argsList);

            ProcessOutputBuffer = null;
            IsVerbose = isVerbose;
            int exitCode = ProcessAccessor.DoProcess(recoveryWriterUsbPath, argsList, RecoveryWriterUsbOutputDataReceived);

            if (exitCode != 0 || !CheckRecoveryWriterUsbSucceeded())
            {
                goto quit;
            }

            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "  Wait 30 sec.");
            Thread.Sleep(30000);

            eStatus = ExitStatus.Success;

        quit:
            Directory.SetCurrentDirectory(currentDirectory);
            return eStatus;
        }

        private static bool ParseRecoveryWriterUsbSerialNumberOutput()
        {
            if (!ProcessOutputBuffer.Contains(RecoveryWriterUsbSuccessString))
            {
                return false;
            }

            string[] lines = ProcessOutputBuffer.Split(new string[]{"\r", "\n"}, StringSplitOptions.RemoveEmptyEntries);
            if (lines[0].Contains(RecoveryWriterUsbSuccessString))
            {
                string parseString = lines[0].Substring(RecoveryWriterUsbSuccessString.Length);
                Regex r = new Regex(@"^\s*(\d+) device");
                Match m = r.Match(parseString);
                if (m.Success)
                {
                    if (!int.TryParse(m.Groups[1].Value, out InSafeModeTargetNum) || InSafeModeTargetNum + 1 > lines.Length)
                    {
                        return false;
                    }
                }
                else
                {
                    return false;
                }
            }
            else
            {
                return false;
            }

            SafeModeSerialNumberList.Clear();
            for (int i = 1; i <= InSafeModeTargetNum; i++)
            {
                string targetString = lines[i].Trim();
                SafeModeSerialNumberList.Add(targetString);
            }

            return true;
        }

        private static bool ParseRecoveryWriterUsbBatteryChargeOutput()
        {
            if (!ProcessOutputBuffer.Contains(RecoveryWriterUsbSuccessString))
            {
                return false;
            }

            string[] lines = ProcessOutputBuffer.Split(new string[] { "\r", "\n" }, StringSplitOptions.RemoveEmptyEntries);
            if (lines[0].Contains(RecoveryWriterUsbSuccessString))
            {
                string parseString = lines[0].Substring(RecoveryWriterUsbSuccessString.Length).Trim();
                if (!int.TryParse(parseString, out BatteryBalue))
                {
                    return false;
                }
                return true;
            }

            return false;
        }

        public static bool IsExistSafeModeSerialNumberList(string targetStringWithDigit)
        {
            foreach (string serialNumber in SafeModeSerialNumberList)
            {
                if (serialNumber.ToUpper() == targetStringWithDigit.ToUpper())
                {
                    if(serialNumber != targetStringWithDigit)
                    {
                        LOG.LogLine(LOG_LEVEL.LOG_WARN, "***********************************************************");
                        LOG.LogLine(LOG_LEVEL.LOG_WARN, " The serial number may be case different. ({0})", serialNumber);
                        LOG.LogLine(LOG_LEVEL.LOG_WARN, " Please contact to Nintendo support office.");
                        LOG.LogLine(LOG_LEVEL.LOG_WARN, "***********************************************************");
                    }
                    return true;
                }
            }
            return false;
        }

        public static int EnumerateSafeModeSerialNumbers(out string[] serialNumbers)
        {
            serialNumbers = SafeModeSerialNumberList.ToArray();
            return InSafeModeTargetNum;
        }

        private static bool ParseConfigurationId1Output(out TargetSpecifier.EdevVersion edevVersion)
        {
            if (ProcessOutputBuffer.Contains(RecoveryWriterUsbSuccessString))
            {
                int stringStart = ProcessOutputBuffer.IndexOf("EDEV_");
                if (stringStart >= 0)
                {
                    if (ProcessOutputBuffer.Substring(stringStart).Contains("EDEV_00_02_00_01"))
                    {
                        edevVersion = TargetSpecifier.EdevVersion.EDEV_EP_2_1;
                        LOG.LogLine(LOG_LEVEL.LOG_INFO2, "EDEV version is {0}", edevVersionDictionary[edevVersion]);
                        return true;
                    }
                    else if (ProcessOutputBuffer.Substring(stringStart).Contains("EDEV_00_02_00_02"))
                    {
                        edevVersion = TargetSpecifier.EdevVersion.EDEV_EP_2_2;
                        LOG.LogLine(LOG_LEVEL.LOG_INFO2, "EDEV version is {0}", edevVersionDictionary[edevVersion]);
                        return true;
                    }
                    else if (ProcessOutputBuffer.Substring(stringStart).Contains("EDEV_00_03_00_00"))
                    {
                        edevVersion = TargetSpecifier.EdevVersion.EDEV_MP;
                        LOG.LogLine(LOG_LEVEL.LOG_INFO2, "EDEV version is {0}", edevVersionDictionary[edevVersion]);
                        return true;
                    }
                }
            }
            edevVersion = TargetSpecifier.EdevVersion.EDEV_ERROR;
            LOG.LogLine(LOG_LEVEL.LOG_ERROR, "EDEV version check failed.");
            return false;
        }

        private static void RecoveryWriterUsbOutputDataReceived(object sender, DataReceivedEventArgs e)
        {
            if (IsVerbose)
            {
                if (e.Data != null && e.Data.Contains(RecoveryWriterUsbSuccessString))
                {
                    Console.WriteLine(ConvertedRecoveryWriterUsbSuccessString + e.Data.Substring(RecoveryWriterUsbSuccessString.Length));
                }
                else if (e.Data != null)
                {
                    Console.WriteLine(e.Data);
                }
            }
            ProcessOutputBuffer += e.Data + "\r\n";
        }

        private static bool CheckRecoveryWriterUsbSucceeded()
        {
            return ProcessOutputBuffer.Contains(RecoveryWriterUsbSuccessString);
        }
    }
}
