﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading;
using Nintendo.InitializeSdev;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;

namespace Nintendo.InitializeEdev
{
    public enum ExitStatus
    {
        Success,
        Failure
    }

    public partial class EdevInitializer
    {
        public EdevInitializeCommandArgument Argument { get; private set; }
        private const int DefaultTimeoutSecond = 600;
        private FirmwareResourceSpecifier firmwareResource;

        public ExitStatus Execute(EdevInitializeCommandArgument arguments)
        {
            Argument = arguments;
            ExitStatus eStatus = ExitStatus.Failure;

            ProcessingProgress progress = ProcessingProgress.GetInstance();
            progress.SetProgress(0);

            if(UsbAccessCheck() != ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }

            progress.SetProgress(5);

            SdevInitializeCommandArgument dummySdevArguments = new SdevInitializeCommandArgument();
            firmwareResource = new FirmwareResourceSpecifier();
            if (firmwareResource.SetVersion(GetVersionString()) == false)
            {
                return ExitStatus.Failure;
            }
            string targetManagerPath = firmwareResource.GetFileNameOfFirm(FirmwareType.Host_TargetManager);
            string oldTargetManagerPath = firmwareResource.GetFileNameOfFirm(FirmwareType.Host_TargetManager_Old);
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(targetManagerPath), targetManagerPath) == false)
            {
                return ExitStatus.Failure;
            }
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(oldTargetManagerPath), oldTargetManagerPath) == false)
            {
                return ExitStatus.Failure;
            }

            if (ExecutePluginsBeforeInitialize() != ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }

            progress.SetProgress(10);

            LOG.LogLine("Stop NintendoTargetManager and check target information.");

            if (StopTargetManager() != ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }

            TargetInnerClass targetKey = new TargetInnerClass(string.Empty, TargetInnerClass.TargetType.TargetInner_Other);

            string targetArgumentString = null;
            EdevInitializeCommandArgument.TargetArgumentType taragetArgumentType = EdevInitializeCommandArgument.TargetArgumentType.TypeOther;
            if (!arguments.ParseTarget(ref targetArgumentString, ref taragetArgumentType))
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Cannot specify target. Please set --target.");
                return ExitStatus.Failure;
            }

            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "EDEV target argument '{0}', type '{1}'", targetArgumentString, taragetArgumentType.ToString());

            TargetManagerAccess.UnregisterTarget(targetManagerPath, targetArgumentString, taragetArgumentType, Argument.Verbose);

            progress.SetProgress(20);

            if (!TargetSpecifier.GetTargetKey(ref targetKey, targetArgumentString, taragetArgumentType, targetManagerPath))
            {
                return ExitStatus.Failure;
            }

            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "EDEV target key is {0} (type:{1})", targetKey.Value != null ? targetKey.Value : string.Empty, targetKey.Type.ToString());

            progress.SetProgress(25);

            eStatus = ExecuteBootSafeMode(ref progress, targetKey, targetManagerPath, oldTargetManagerPath);
            if (eStatus != ExitStatus.Success)
            {
                goto quit;
            }

            progress.SetProgress(40);

            TargetSpecifier.EdevVersion edevVersion = TargetSpecifier.EdevVersion.EDEV_NONE;
            bool isNewSafeMode;
            bool isSerialIdFound;
            eStatus = CheckNewSafeMode(out isNewSafeMode, targetKey, targetManagerPath, out isSerialIdFound);
            if (eStatus != ExitStatus.Success)
            {
                goto quit;
            }

            if (isSerialIdFound)
            {
                eStatus = GetEdevVersionAndUpdateBootImageInSafeMode(out edevVersion, isNewSafeMode, targetKey, ref progress);
            }
            else
            {
                // SystemUpdater待機中の可能性があるのでSafeModeのSerialNumberが見つからなくても処理を継続する
                eStatus = GetEdevVersionInNormalMode(out edevVersion, targetKey, targetManagerPath, ref progress);
            }
            if (eStatus != ExitStatus.Success)
            {
                goto quit;
            }

            eStatus = ValidateEdevVersion(firmwareResource);
            if (eStatus != ExitStatus.Success)
            {
                goto quit;
            }

            progress.SetProgress(55);

            string systemUpdaterPath = firmwareResource.GetFileNameOfFirm(FirmwareType.Firmware_SystemUpdaterProcess);
#if SYSTEM_UPDATE_IMAGE
            string systemImagePath = firmwareResource.GetFileNameOfFirm(FirmwareType.Firmware_SystemImage);
#endif
            string safeModeUpdaterPath = firmwareResource.GetFileNameOfFirm(FirmwareType.Firmware_SafeModeUpdaterProcess);
#if SYSTEM_UPDATE_IMAGE
            string safeModeImagePath = firmwareResource.GetFileNameOfFirm(FirmwareType.Firmware_SafeModeImage);
#endif
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(systemUpdaterPath), systemUpdaterPath) == false)
            {
                return ExitStatus.Failure;
            }
#if SYSTEM_UPDATE_IMAGE
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(systemImagePath), systemImagePath) == false)
            {
                return ExitStatus.Failure;
            }
#endif
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(safeModeUpdaterPath), safeModeUpdaterPath) == false)
            {
                return ExitStatus.Failure;
            }
#if SYSTEM_UPDATE_IMAGE
            if (FirmwareResourceSpecifier.CheckFilePath(nameof(safeModeImagePath), safeModeImagePath) == false)
            {
                return ExitStatus.Failure;
            }
#endif

#if SYSTEM_UPDATE_IMAGE
            eStatus = UpdateSafeMode(systemUpdaterPath, targetKey, safeModeImagePath, targetManagerPath);
#else
            eStatus = UpdateSafeMode(safeModeUpdaterPath, targetKey, targetManagerPath);
#endif
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Update safe mode failed.");
                goto quit;
            }

            progress.SetProgress(75);

            eStatus = DisconnectTarget(targetKey, targetManagerPath);
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_WARN, "Failed to disconnect target. ({0})", targetKey);
            }

#if SYSTEM_UPDATE_IMAGE
            eStatus = UpdateSystemImage(systemUpdaterPath, targetKey, systemImagePath, targetManagerPath);
#else
            eStatus = UpdateSystemImage(systemUpdaterPath, targetKey, targetManagerPath);
#endif
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Update system image failed.");
                goto quit;
            }

            eStatus = ActivateTarget(targetKey, targetManagerPath);
            if (eStatus != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_WARN, "Failed to activate target. ({0})", targetKey);
            }

            StopTargetManagerForLast();

            if (ExecutePluginsAfterInitialize(targetKey, edevVersion) != ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }

            LOG.LogLine("Edev initialization SUCCEEDED.");

            progress.SetProgress(100);

            eStatus = ExitStatus.Success;

        quit:

            return eStatus;
        }
    }
}
