﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.ControlTarget;
using Nintendo.ControlTarget.ConsoleShell;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;

namespace Nintendo.InitializeEdev
{
    public enum BootSafeModeResult
    {
        Success,
        NotFound,
        Failure
    }

    public class BootSafeMode
    {
        public static void StartOldTargetManager(FileInfo exePath)
        {
            if (!exePath.Exists)
            {
                throw new Exception("Executable file was not found.");
            }

            var process = new Process();

            process.StartInfo.FileName = exePath.FullName;
            process.StartInfo.Arguments = "-tasktray";

            process.Start();
        }

        public static bool BootSafeModeFor14483(FileInfo exePath, string targetKeyString)
        {
            if (!exePath.Exists)
            {
                throw new Exception("Executable file was not found.");
            }

            var process = new Process();

            process.StartInfo.FileName = exePath.FullName;
            process.StartInfo.Arguments = "boot-safemode --target " + targetKeyString;
            process.StartInfo.RedirectStandardOutput = true;
            process.StartInfo.UseShellExecute = false;

            process.Start();
            process.WaitForExit();

            return (process.ExitCode == 0);
        }

        public static bool UnregisterTargetFor14483(FileInfo exePath, string targetKeyString)
        {
            if (!exePath.Exists)
            {
                throw new Exception("Executable file was not found.");
            }

            var process = new Process();

            process.StartInfo.FileName = exePath.FullName;
            process.StartInfo.Arguments = "unregister --target " + targetKeyString;
            process.StartInfo.RedirectStandardOutput = true;
            process.StartInfo.UseShellExecute = false;

            process.Start();
            process.WaitForExit();

            return (process.ExitCode == 0);
        }

        public static BootSafeModeResult Execute(string serialNumberString, string targetManagerDirectory, string oldTargetManagerDirectory, bool isVerbose)
        {
            ControlTarget.Tmapi.RegisteredTargetInfo targetInfo;
            string targetString;
            if (serialNumberString.ToUpper() != "USB")
            {
                Nintendo.InitializeSdev.SerialNumber serial = new Nintendo.InitializeSdev.SerialNumber(serialNumberString);
                if (serial.GetString() == null)
                {
                    return BootSafeModeResult.Failure;
                }
                targetString = serial.GetStringWithCheckDigit();
            }
            else
            {
                targetString = "usb";
            }

            using (var tmapiAccessor = new TargetManagerAccessor())
            {
                tmapiAccessor.SetExtraTargetManagerDirectory(targetManagerDirectory);
                tmapiAccessor.SetStartTrayIcon(true);
                tmapiAccessor.EnsureStart();

                // MEMO: TM起動直後の動作対策のため調整しました (2017.07.26)
                LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Wait 5 sec for preparing to find.");
                System.Threading.Thread.Sleep(5000);

                ControlTarget.Tmapi.TargetHandle targetHandle;
                try
                {
                    targetHandle = tmapiAccessor.FindTarget(targetString);
                }
                catch (Exception)
                {
                    return BootSafeModeResult.NotFound;
                }

                targetInfo = tmapiAccessor.FindRegisteredTarget(targetHandle);
            }

            String[] tmaVersion = targetInfo.GetTmaVersion().Split('.');
            var tmaBuildVersion = tmaVersion[tmaVersion.Length - 1];
            if (string.IsNullOrEmpty(tmaBuildVersion))
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Failed to get the version.");
                return BootSafeModeResult.Failure;
            }

            if (tmaBuildVersion == "14483")
            {
                // 最新の TargetManager で追加されたターゲット情報は過去の TargetManager で互換性が保証されないので Unregsiter する
                using (var tmapiAccessor = new TargetManagerAccessor())
                {
                    tmapiAccessor.SetExtraTargetManagerDirectory(targetManagerDirectory);
                    tmapiAccessor.SetStartTrayIcon(true);
                    tmapiAccessor.EnsureStart();

                    tmapiAccessor.Unregister(tmapiAccessor.FindTarget(targetString));
                }

                if (InitializeSdev.ProcessAccessor.StopTargetManager() != Nintendo.InitializeSdev.ExitStatus.Success)
                {
                    return BootSafeModeResult.Failure;
                }

                StartOldTargetManager(new FileInfo(Path.Combine(oldTargetManagerDirectory, @"NintendoTargetManager.exe")));

                if (!BootSafeModeFor14483(new FileInfo(Path.Combine(oldTargetManagerDirectory, @"Console.exe")), targetString))
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Failed to boot safemode.");
                    return BootSafeModeResult.Failure;
                }

                if (!UnregisterTargetFor14483(new FileInfo(Path.Combine(oldTargetManagerDirectory, @"Console.exe")), targetString))
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Failed to unregister target.");
                    return BootSafeModeResult.Failure;
                }

                if (InitializeSdev.ProcessAccessor.StopTargetManager() != Nintendo.InitializeSdev.ExitStatus.Success)
                {
                    return BootSafeModeResult.Failure;
                }
            }
            else
            {
                if (InitializeSdev.ProcessAccessor.StopTargetManager() != Nintendo.InitializeSdev.ExitStatus.Success)
                {
                    return BootSafeModeResult.Failure;
                }

                using (var tmapiAccessor = new TargetManagerAccessor())
                {
                    tmapiAccessor.SetExtraTargetManagerDirectory(targetManagerDirectory);
                    tmapiAccessor.SetStartTrayIcon(true);
                    tmapiAccessor.SetStartForInitialize(true);
                    tmapiAccessor.EnsureStart();

                    // MEMO: TM起動直後の動作対策のため調整しました (2017.07.26)
                    LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Wait 5 sec for preparing to connect.");
                    System.Threading.Thread.Sleep(5000);

                    var info = tmapiAccessor.FindTargetInfo(targetString);

                    tmapiAccessor.ConnectTarget(info.GetTargetHandle());

                    System.Net.IPEndPoint endPoint;
                    try
                    {
                        endPoint = tmapiAccessor.FindService("iywys@$csForRunnerTools", info.GetTargetHandle());
                    }
                    catch (HtcServiceNotFound)
                    {
                        try
                        {
                            endPoint = tmapiAccessor.FindService("@csForRunnerTools", info.GetTargetHandle());
                        }
                        catch (HtcServiceNotFound)
                        {
                            throw new Exception("Found no shell service.");
                        }
                    }

                    var csAccessor = new ConsoleShellAccessor(new ConsoleShellMessenger(endPoint));

                    csAccessor.BootSafeMode();
                }
            }

            return BootSafeModeResult.Success;

        }
    }
}
