﻿using Nintendo.ControlTarget;
using System;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace Nintendo.ControlTarget
{
    public enum TargetProgramKind
    {
        InstalledApplicationId,
        InstalledSystemApplicationId,
        GameCardApplication,
        NspDirectory,
        NspFile
    }

    public class TargetProgramInfo
    {
        private static readonly Regex ApplicationIdPattern = new Regex("^0x[0-9A-Fa-f]{16}$");
        private static readonly Regex SystemApplicationIdPattern = new Regex("^system:(0x[0-9A-Fa-f]{16})$");
        private static readonly Regex GameCardPattern = new Regex("^gamecard$");

        public string Name { get; private set; }

        public TargetProgramKind Kind { get; private set; }

        public TargetProgramInfo(string name, bool isNcaSupported)
        {
            Name = name;
            Kind = GetKind(name, isNcaSupported);
        }

        public FileInfo NspFile
        {
            get
            {
                ValidateKind(TargetProgramKind.NspFile);
                return new FileInfo(Name);
            }
        }

        public DirectoryInfo NspDirectory
        {
            get
            {
                ValidateKind(TargetProgramKind.NspDirectory);
                return new DirectoryInfo(Name);
            }
        }

        public ulong InstalledApplicationId
        {
            get
            {
                ValidateKind(TargetProgramKind.InstalledApplicationId);
                return Convert.ToUInt64(Name, 16);
            }
        }

        public ulong InstalledSystemApplicationId
        {
            get
            {
                ValidateKind(TargetProgramKind.InstalledSystemApplicationId);
                var match = SystemApplicationIdPattern.Match(Name);
                return Convert.ToUInt64(match.Groups[1].Value, 16);
            }
        }

        private void ValidateKind(TargetProgramKind expected)
        {
            if (expected != Kind)
            {
                throw new InvalidDataException($"Unmatched kind: expected({expected}) != actual({Kind})");
            }
        }

        public static TargetProgramKind GetKind(string name, bool isNcaSupported)
        {
            string[] fileExtensions = isNcaSupported ? new string[] { ".nca", ".nsp" } : new string[] { ".nsp" };
            string[] directoryExtensions = new string[] { ".nspd" };

            if (ApplicationIdPattern.IsMatch(name))
            {
                return TargetProgramKind.InstalledApplicationId;
            }
            else if(SystemApplicationIdPattern.IsMatch(name))
            {
                return TargetProgramKind.InstalledSystemApplicationId;
            }
            else if(GameCardPattern.IsMatch(name))
            {
                return TargetProgramKind.GameCardApplication;
            }
            else if (File.Exists(name))
            {
                var fileInfo = new FileInfo(name);

                if (!fileExtensions.Contains(fileInfo.Extension))
                {
                    throw new InvalidUsage(
                        string.Format("Unsuppoted extension(file): {0}, supported: {1}",
                            name, string.Join(", ", fileExtensions)));
                }

                return TargetProgramKind.NspFile;
            }
            else if (Directory.Exists(name))
            {
                var directoryInfo = new DirectoryInfo(name);

                if (!directoryExtensions.Contains(directoryInfo.Extension))
                {
                    throw new InvalidUsage(
                        string.Format("Unsuppoted extension(direcotry): {0}, supported: {1}",
                            name, string.Join(", ", directoryExtensions)));
                }

                return TargetProgramKind.NspDirectory;
            }
            else
            {
                throw new InvalidUsage(string.Format("Found no programs: {0}", name));
            }
        }
    }
}
