﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Threading.Tasks;
using System.Threading.Tasks.Dataflow;
using System.Diagnostics;
using System.Net.Sockets;
using System.Net;

namespace Nintendo.ControlTarget.ConsoleShell
{
    public abstract class IConsoleShellMessenger
    {
        public ISourceBlock<IConsoleShellResponse> ReceivePort { get; protected set; }
        public ITargetBlock<ConsoleShellRequest> SendPort { get; protected set; }
    }

    public class ConsoleShellMessenger : IConsoleShellMessenger, IDisposable
    {
        private Socket NetworkSocket;
        private NetworkStream NetworkStream;
        private Stream SendStream;
        private Stream ReceiveStream;
        private BufferBlock<IConsoleShellResponse> ReceiveBuffer;

        public ConsoleShellMessenger(Stream sendStream, Stream receiveStream)
        {
            SendStream = sendStream;
            ReceiveStream = receiveStream;
            ReceiveBuffer = new BufferBlock<IConsoleShellResponse>();

            SendPort = new ActionBlock<ConsoleShellRequest>(request => { this.SendRequest(request); });

            ReceivePort = ReceiveBuffer;
        }

        public ConsoleShellMessenger(IPEndPoint endPoint)
        {
            NetworkSocket = NetworkUtility.CreateConnectedSocket(endPoint);
            NetworkStream = new NetworkStream(NetworkSocket);

            SendStream = NetworkStream;
            ReceiveStream = NetworkStream;
            ReceiveBuffer = new BufferBlock<IConsoleShellResponse>();

            SendPort = new ActionBlock<ConsoleShellRequest>(request => { this.SendRequest(request); });

            ReceivePort = ReceiveBuffer;

            RunServer();
        }

        public void RunServer()
        {
            Task.Run(() => {
                try
                {
                    this.ProcessingReceiveStream();
                }
                catch (Exception e)
                {
                    Console.Error.WriteLine("Error in console shell: {0}", e.Message);
                }
            });
        }

        public void SendRequest(ConsoleShellRequest request)
        {
            Trace.WriteLine($"SendRequest: {request.RequestExpression}");
            SendStream.Write(request.RequestData, 0, request.RequestData.Length);
            SendStream.Flush();
        }

        public void ProcessingReceiveStream()
        {
            while (true)
            {
                byte[] headerBinary = new byte[CommandResponseHeader.HeaderSize];
                int readSize = ReceiveStream.Read(headerBinary, 0, CommandResponseHeader.HeaderSize);

                if (readSize != CommandResponseHeader.HeaderSize)
                {
                    throw new Exception("Disconnected");
                }

                CommandResponseHeader header = new CommandResponseHeader(headerBinary);
                Trace.WriteLine($"ReceiveHeader: RequestId={header.RequestId,0:X8}, CommandId={header.CommandId}, bodySize: {header.BodySize}");

                int totalReadSize = 0;
                MemoryStream ms = new MemoryStream();
                while (totalReadSize < header.BodySize)
                {
                    const int BufferSize = 32 * 1024;
                    byte[] buffer = new byte[BufferSize];
                    int readBodySize = ReceiveStream.Read(buffer, 0, Math.Min((int)header.BodySize - totalReadSize, BufferSize));
                    if (readBodySize == 0)
                    {
                        break;
                    }
                    ms.Write(buffer, 0, readBodySize);
                    totalReadSize += readBodySize;
                }

                if (totalReadSize != header.BodySize)
                {
                    throw new Exception("Disconnected");
                }
                ms.Flush();
                ms.Position = 0;

                byte[] bodyBinary = new byte[header.BodySize];
                ms.Read(bodyBinary, 0, (int)header.BodySize);
                try
                {
                    var response = IConsoleShellResponse.MakeResponse(header, bodyBinary);
                    Trace.WriteLine(response.ResponseExpression);
                    ReceiveBuffer.Post(response);
                }
                catch (Exception e)
                {
                    Trace.WriteLine($"[ERROR] {e.ToString()}");
                }

                Trace.WriteLine("Processed response");
            }
        }

        public void Dispose()
        {
            if (NetworkStream != null)
            {
                NetworkStream.Dispose();
            }

            if (NetworkSocket != null)
            {
                NetworkSocket.Dispose();
            }
        }
    }
}
