﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.IO;
using System.Threading.Tasks;
using System.Threading.Tasks.Dataflow;
using System.Diagnostics;

namespace Nintendo.ControlTarget.ConsoleShell
{
    public enum ResponseProcessingStatus
    {
        Continue,
        Exit
    }

    public class ConsoleShellAccessor : IDisposable
    {
        private IConsoleShellMessenger Messenger;
        private Dictionary<ulong, Func<IConsoleShellResponse, ResponseProcessingStatus>> ResponseProcessors = new Dictionary<ulong, Func<IConsoleShellResponse, ResponseProcessingStatus>>();
        private ActionBlock<IConsoleShellResponse> ResponseProcessor;

        public ConsoleShellAccessor(IConsoleShellMessenger messenger)
        {
            Messenger = messenger;

            ResponseProcessor = new ActionBlock<IConsoleShellResponse>(response =>
            {
                this.ProcessResponse(response);
            });

            Messenger.ReceivePort.LinkTo(ResponseProcessor);
        }

        public void ProcessResponse(IConsoleShellResponse response)
        {
            if (ResponseProcessors.ContainsKey(response.Header.RequestId))
            {
                var result = ResponseProcessors[response.Header.RequestId](response);

                if (result == ResponseProcessingStatus.Exit)
                {
                    ResponseProcessors.Remove(response.Header.RequestId);
                }
            }
            else
            {
                Trace.WriteLine($"Found no processor: {response.ResponseExpression}");
            }
        }

        public void RegisterResponseProcessor(ulong requestId, Func<IConsoleShellResponse, ResponseProcessingStatus> responseProcessor)
        {
            if (ResponseProcessors.ContainsKey(requestId))
            {
                throw new Exception("Request processor is already registered: " + requestId.ToString());
            }

            ResponseProcessors[requestId] = responseProcessor;
        }

        public void UnregisterResponseProcessor(ulong requestId)
        {
            if (!ResponseProcessors.ContainsKey(requestId))
            {
                throw new Exception("Found no request processor: " + requestId.ToString());
            }

            ResponseProcessors.Remove(requestId);
        }

        public void SendRequestAsync(IConsoleShellCommand command, Func<IConsoleShellResponse, ResponseProcessingStatus> responseProcessor)
        {
            var request = command.MakeRequest();
            var requestId = request.RequestId;

            RegisterResponseProcessor(requestId, responseProcessor);

            Messenger.SendPort.Post(request);
        }

        public void SendRequest(IConsoleShellCommand command, Func<IConsoleShellResponse, ResponseProcessingStatus> responseProcessor = null)
        {
            var commandResponseCompletion = new TaskCompletionSource<Exception>();
            var additionalProcessorCompletion = new TaskCompletionSource<bool>();

            SendRequestAsync(command, response =>
            {
                response.ProcessIf<SuccessResponse>(successResponse =>
                {
                    commandResponseCompletion.SetResult(null);
                });

                response.ProcessIf<FailureResponse>(failureResponse =>
                {
                    string message = MakeErrorMassage(command, failureResponse);
                    commandResponseCompletion.SetResult(new Exception(message));
                });

                if (responseProcessor == null)
                {
                    if (commandResponseCompletion.Task.IsCompleted)
                    {
                        return ResponseProcessingStatus.Exit;
                    }
                    else
                    {
                        return ResponseProcessingStatus.Continue;
                    }
                }
                else
                {
                    var processorResponse = responseProcessor(response);

                    if( processorResponse == ResponseProcessingStatus.Exit)
                    {
                        additionalProcessorCompletion.SetResult(true);
                    }

                    if (commandResponseCompletion.Task.IsCompleted && additionalProcessorCompletion.Task.IsCompleted)
                    {
                        return ResponseProcessingStatus.Exit;
                    }
                    else
                    {
                        return ResponseProcessingStatus.Continue;
                    }
                }
            });

            RetryUtility.Do(() =>
            {
                commandResponseCompletion.Task.Wait(TimeSpan.FromSeconds(5));

                Trace.WriteLine($"Wait {command.CommandName}");

                if (!commandResponseCompletion.Task.IsCompleted)
                {
                    throw new Exception("Can't communicate to shell");
                }

                Trace.WriteLine($"Returned from {command.CommandName}");
            },
            e =>
            {
            },
            30,
            TimeSpan.FromSeconds(5));

            if (commandResponseCompletion.Task.Result != null)
            {
                throw commandResponseCompletion.Task.Result;
            }
        }

        private static string MakeErrorMassage(IConsoleShellCommand command, FailureResponse failureResponse)
        {
            var prefix = $"Command({command.GetType().Name}) failed(nn::Result=0x{failureResponse.Result:X8})";

            switch (failureResponse.Result)
            {
                case 0x0007CC02:
                    return $"{prefix}: Built with an older version SDK. Must be rebuilt.";
                case 0x00000A09:
                    return $"{prefix}: Invalid NSO file configuration in the code directory.";
                default:
                    return $"{prefix}";
            }
        }

        public Task<int> LaunchApplication(TargetProgramInfo path, string[] arguments)
        {
            return LaunchApplication(path, arguments, new LaunchApplicationOption());
        }

        public Task<int> LaunchApplication(TargetProgramInfo path, string[] arguments, LaunchApplicationOption option)
        {
            var command = MakeLaunchCommand(path, arguments, option);

            var applicationExitResponseCompletion = new TaskCompletionSource<int>();

            SendRequest(command, response =>
            {
                var status = ResponseProcessingStatus.Continue;

                response.ProcessIf<ApplicationExitResponse>(exitResponse =>
                {
                    applicationExitResponseCompletion.SetResult(exitResponse.ExitStatus);

                    status = ResponseProcessingStatus.Exit;
                });

                response.ProcessIf<JitDebugNotificationResponse>(jitDebugResponse =>
                {
                    applicationExitResponseCompletion.SetResult(jitDebugResponse.ExitStatus);

                    status = ResponseProcessingStatus.Exit;
                });

                return status;
            });

            return applicationExitResponseCompletion.Task;
        }

        private static IConsoleShellCommand MakeLaunchCommand(TargetProgramInfo program, string[] arguments, LaunchApplicationOption option)
        {
            var sentArgument = (new string[] { program.Name }).Concat(arguments == null ? new string[0] : arguments).ToArray();
            switch (program.Kind)
            {
                case TargetProgramKind.InstalledApplicationId:
                    return new LaunchInstalledApplicationCommand(program.InstalledApplicationId, sentArgument, option);
                case TargetProgramKind.InstalledSystemApplicationId:
                    return new LaunchInstalledSystemApplicationCommand(program.InstalledSystemApplicationId, sentArgument);
                case TargetProgramKind.GameCardApplication:
                    return new LaunchGameCardApplicationCommand(sentArgument, option);
                case TargetProgramKind.NspDirectory:
                    {
                        string stringPath = program.NspDirectory.FullName + "/program0.ncd/";
                        sentArgument[0] = program.NspDirectory.FullName;
                        return new LaunchApplicationCommand(stringPath, sentArgument, option);
                    }
                case TargetProgramKind.NspFile:
                    {
                        string stringPath = program.NspFile.FullName;
                        sentArgument[0] = program.NspFile.FullName;
                        return new LaunchApplicationCommand(stringPath, sentArgument, option);
                    }
                default:
                    throw new Exception("Found no file or directory: " + program);
            }
        }

        public void TerminateAllApplications()
        {
            try
            {
                SendRequest(new TerminateAppicationCommand());
            }
            catch (Exception e)
            {
                Trace.WriteLine(e.Message);
            }

            try
            {
                SendRequest(new TerminateProcessCommand());
            }
            catch (Exception e)
            {
                Trace.WriteLine(e.Message);
            }
        }

        public void SaveScreenShot(string path)
        {
            try
            {
                SendRequest(new SaveScreenShotCommand(), response =>
                {
                    var status = ResponseProcessingStatus.Continue;
                    response.ProcessIf<ScreenShotResponse>(screenShotResponse =>
                    {
                        screenShotResponse.Save(path);
                        status = ResponseProcessingStatus.Exit;
                    });
                    return status;
                });
            }
            catch (Exception e)
            {
                Trace.WriteLine(e.Message);
            }
        }

        public void BootSafeMode()
        {
            try
            {
                SendRequest(new SetSafeModeCommand());
                SendRequest(new RebootCommand());
            }
            catch (Exception e)
            {
                Trace.WriteLine(e.Message);
            }
        }

        public void Reset()
        {
            try
            {
                SendRequest(new RebootCommand());
            }
            catch (Exception e)
            {
                Trace.WriteLine(e.Message);
            }
        }

        public void Shutdown()
        {
            try
            {
                SendRequest(new ShutdownCommand());
            }
            catch (Exception e)
            {
                Trace.WriteLine(e.Message);
            }
        }

        public void ControlVirtualTemperature(bool isEnable, int temperature)
        {
            try
            {
                SendRequest(new ControlVirtualTemperatureCommand(isEnable, temperature));
            }
            catch (Exception e)
            {
                Trace.WriteLine(e.Message);
            }
        }

        public void Dispose()
        {
            ResponseProcessor.Complete();
            ResponseProcessor.Completion.Wait();
        }
    }
}
