﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Diagnostics;

using Phidgets;

namespace Nintendo.ControlTarget
{
    public class PhidgetUtility
    {
        public static void EnableVerboseLog()
        {
            Phidget.enableLogging(Phidget.LogLevel.PHIDGET_LOG_VERBOSE, null);
        }

        public static void EnsureOutput(Phidgets.InterfaceKit ifKit, int index, bool state)
        {
            Phidget.log(Phidget.LogLevel.PHIDGET_LOG_VERBOSE, null, String.Format("outputs[{0}] -> {1}", index, state));
            ifKit.outputs[index] = state;

            const int retryMax = 10;
            int retryCount = 0;
            while (ifKit.outputs[index] != state)
            {
                retryCount++;
                Phidget.log(Phidget.LogLevel.PHIDGET_LOG_VERBOSE, null, String.Format("Waiting for state change ({0})", retryCount));
                Task.Delay(10).Wait();
                if(retryCount > retryMax)
                {
                    throw new PhidgetException("Checking state retry count is exceeded.", PhidgetException.ErrorType.PHIDGET_ERR_TIMEOUT);
                }
            }
        }

        public static void SendPulse(int index)
        {
            SendPulse(index, 0, TimeSpan.FromMilliseconds(500));
        }

        public static void SendPulse(int index, int serialNumber, TimeSpan timeOut)
        {
            var ifKit = new InterfaceKit();

            try
            {
                if (serialNumber == 0)
                {
                    ifKit.open();
                }
                else
                {
                    ifKit.open(serialNumber);
                }

                ifKit.waitForAttachment((int)timeOut.TotalMilliseconds);
            }
            catch
            {
                Console.Error.WriteLine("Found no phidget device");
                throw;
            }

            ifKit.outputs[index] = true;
            Task.Delay(100).Wait();
            ifKit.outputs[index] = false;

            ifKit.close();
        }

        public static void SendSignals(Action<InterfaceKit> action)
        {
            SendSignals(action, 0, TimeSpan.FromMilliseconds(500));
        }

        public static void SendSignals(Action<InterfaceKit> action, int serialNumber, TimeSpan timeOut)
        {
            var ifKit = new InterfaceKit();

            try
            {
                if (serialNumber == 0)
                {
                    ifKit.open();
                }
                else
                {
                    ifKit.open(serialNumber);
                }

                ifKit.waitForAttachment((int)timeOut.TotalMilliseconds);
            }
            catch
            {
                Console.Error.WriteLine("Found no phidget device");
                throw;
            }

            action(ifKit);

            ifKit.close();
        }
    }
}
