﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace Nintendo.Bridge
{
    public class Telnet
    {
        public const int DEFAULT_PORT = 23;
        public const int TARGET_PORT = 10023;
        const int DEFAULT_TIMEOUT = 8000;
        const string END_CONSOLE_PROMPT = "# ";
        const string START_CONSOLE_PROMPT = "root@";

        public bool Verbose { get; set; }
        public bool IsConnected { get; set; }

        AutoResetEvent evtRecv = null;

        NetworkStream ns = null;
        TcpClient tcp = null;

        string log = null;

        public Telnet()
        {
            Verbose = false;
            IsConnected = false;
            log = "";
            evtRecv = new AutoResetEvent(false);
        }

        public bool Connect(string address, int port = DEFAULT_PORT)
        {
            int retry = 3;

            while (retry-- > 0)
            {
                try
                {
                    tcp = new TcpClient(address, port);
                    ns = tcp.GetStream();

                    Thread thread = new Thread(recv);
                    thread.IsBackground = true;
                    thread.Start();

                    IsConnected = true;
                    return true;
                }
                catch (Exception e)
                {
                    if (Verbose)
                    {
                        Console.WriteLine(e);
                    }
                }
            }

            IsConnected = false;
            return false;
        }

        public void Exit()
        {
            IsConnected = false;

            if (ns != null)
            {
                ns.Close();
            }

            if (tcp != null)
            {
                tcp.Close();
            }
        }

        void recv()
        {
            byte[] resBytes = new byte[65536];

            while (true)
            {
                try
                {
                    do
                    {
                        int resSize = ns.Read(resBytes, 0, resBytes.Length);
                        if (resSize == 0)
                        {
                            IsConnected = false;
                            break;
                        }

                        string diff = Encoding.UTF8.GetString(resBytes, 0, resSize);

                        log += diff;

                        if (Verbose)
                        {
                            Console.Write(diff);
                        }
                    }
                    while (ns.DataAvailable);

                    evtRecv.Set();
                }
                catch (Exception e)
                {
                    IsConnected = false;
                    if (Verbose)
                    {
                        Console.WriteLine(e);
                    }
                    break;
                }
            }
        }

        public void Wait(string str, int timeout = DEFAULT_TIMEOUT)
        {
            while (true)
            {
                if (evtRecv.WaitOne(timeout) == false)
                {
                    throw new TimeoutException();
                }

                if (log.EndsWith(str))
                {
                    break;
                }
            }
        }

        public void Send(string line)
        {
            byte[] sendBytes = Encoding.UTF8.GetBytes(line + '\n');
            ns.Write(sendBytes, 0, sendBytes.Length);
            ns.Flush();
        }

        public bool Login()
        {
            try
            {
                Wait("login: ");
                Send("root");
                Wait("Password: ");
                Send("root");
                Wait("# ");
                Send("set +o emacs");
                Wait("# ");
            }
            catch (Exception e)
            {
                if (Verbose)
                {
                    Console.WriteLine(e);
                }
                Exit();
                return false;
            }

            return true;
        }

        public string WaitCommand(string command, int timeout = DEFAULT_TIMEOUT)
        {
            if (!IsConnected)
            {
                Console.WriteLine("Not connected yet.\n");
                return null;
            }

            Send(command);

            try
            {
                Wait(END_CONSOLE_PROMPT, timeout);
            }
            catch (Exception e)
            {
                if (Verbose)
                {
                    Console.WriteLine(e);
                }
                return log;
            }

            string response = null;

            response = log;

            try
            {
                response = response.Replace("\r\r\n", "");
                response = response.Substring(response.LastIndexOf(command) + command.Length + "\r\n".Length);
                response = response.Substring(0, response.LastIndexOf(START_CONSOLE_PROMPT));
            }
            catch (Exception e)
            {
                if (Verbose)
                {
                    Console.WriteLine(e);
                }
                return log;
            }

            return response;
        }
    }
}
