﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace Nintendo.Bridge
{
    public class BridgeInfo
    {
        public bool Verbose { get; set; }

        public bool Valid { get { return valid; } }
        public string Address { get { return address; } }
        public string Name { get { return name; } }
        public UInt32 Port { get { return port; } }
        public string MacAddress { get { return mac; } }
        public UInt32 Status { get { return status; } }
        public UInt32 Version { get { return version; } }
        public byte HwVersion { get { return hwVersion; } }
        public string SerialNumber { get { return serial; } }
        public string Host { get { return host; } }

        bool valid;         // Vaildity
        string address;     // Bridge IP address
        string name;        // Bridge name
        UInt32 port;        // The port number to transfer data
        string mac;         // Bridge MAC address in EEPROM
        UInt32 status = 0;  // Target Status
        UInt32 version;     // Protocol version
        byte hwVersion;     // encoded hardware id of the bridge/SDEV
        string serial;      // Target serial number in EEPROM
        string host;        // Latest connected host machine name

        public BridgeInfo()
        {
            Verbose = false;
            valid = false;
        }

        private void getInfo()
        {
            byte[] eeprom = new byte[1024];

            Telnet telnet = new Telnet();
            if (!telnet.Connect(address))
            {
                valid = false;
                return;
            }

            telnet.Login();
            string data = telnet.WaitCommand("hexdump -v -e '1/1 \"%02x \"' /sys/bus/i2c/devices/0-0054/eeprom");
            host = telnet.WaitCommand(@"if [ -r /var/run/hostname ]; then cat /var/run/hostname; fi");
            telnet.Exit();

            try
            {
                data = data.TrimEnd(' ');
                string[] values = data.Split(' ');

                if (values.Length == eeprom.Length)
                {
                    for (int i = 0; i < eeprom.Length; i++)
                    {
                        eeprom[i] = Convert.ToByte(values[i], 16);
                    }

                    name = Encoding.ASCII.GetString(eeprom, 0x80, 64);
                    name = name.TrimEnd('\0');
                    hwVersion = eeprom[7];
                    serial = Encoding.ASCII.GetString(eeprom, 0x208, 16);
                    serial = SerialNumber.TrimEnd('\0');
                    port = BitConverter.ToUInt16(eeprom, 54);
                    mac = string.Format
                        (
                            "{0:X2}:{1:X2}:{2:X2}:{3:X2}:{4:X2}:{5:X2}",
                            eeprom[0], eeprom[1], eeprom[2], eeprom[3], eeprom[4], eeprom[5]
                        );
                }

                if (host != null)
                {
                    host = host.Replace("\r\n", "");
                }

                valid = true;
            }
            catch (Exception e)
            {
                if (Verbose)
                {
                    Console.WriteLine(e);
                }
                valid = false;
            }
        }

        public void SetInfo(IPEndPoint endPoint, byte[] info)
        {
            try
            {
                address = endPoint.Address.ToString();
                byte nameLen = info[7];
                byte[] namebytes = new byte[nameLen];
                Array.Copy(info, 16, namebytes, 0, nameLen);
                name = Encoding.ASCII.GetString(namebytes);
                mac = string.Format
                            (
                                "{0:X2}:{1:X2}:{2:X2}:{3:X2}:{4:X2}:{5:X2}",
                                info[1], info[2], info[3], info[4], info[5], info[6]
                            );
                status = 0;  // Targe Status
                version = BitConverter.ToUInt32(info, 12);
                hwVersion = info[238];
                byte[] serialbytes = new byte[20];
                Array.Copy(info, 239, serialbytes, 0, serialbytes.Length);
                serial = Encoding.ASCII.GetString(serialbytes);
                serial = serial.TrimEnd('\0');
                byte[] hostbytes = new byte[40];
                Array.Copy(info, 198, hostbytes, 0, hostbytes.Length);
                host = Encoding.ASCII.GetString(hostbytes);
                host = host.TrimEnd('\0');
            }
            catch (Exception e)
            {
                if (Verbose)
                {
                    Console.WriteLine(e);
                }
            }
        }

        public void GetInfo(string address, bool async = false)
        {
            this.address = address;

            if (async)
            {
                Thread thread = new Thread(getInfo);
                thread.IsBackground = true;
                thread.Start();
            }
            else
            {
                getInfo();
            }
        }

        public bool GetRecoveryHash(string address, out string sion, out string tics)
        {
            sion = null;
            tics = null;

            Telnet telnet = new Telnet();
            if (!telnet.Connect(address))
            {
                return false;
            }

            bool result = true;

            try
            {
                telnet.Login();
                telnet.WaitCommand("nanddump /dev/flash/boot2 -s 0xd00000 -l 0x20000 -f /tmp/boot2.bin");
                sion = telnet.WaitCommand(@"strings /tmp/boot2.bin | grep firmware_hash | sed -e 's/.*firmware_hash=\([^ ]*\).*/\1/'");
                sion = sion.Replace("\r\n", "");
                tics = telnet.WaitCommand(@"strings /tmp/boot2.bin | grep tics_hash | sed -e 's/.*tics_hash=\([^ ]*\).*/\1/'");
                tics = tics.Replace("\r\n", "");
                telnet.WaitCommand("rm -f /tmp/boot2.bin");
            }
            catch (Exception e)
            {
                if (Verbose)
                {
                    Console.WriteLine(e);
                }
                result = false;
            }

            telnet.Exit();
            return result;
        }

        public bool GetHwVersion(string address, out string version)
        {
            version = null;

            Telnet telnet = new Telnet();
            if (!telnet.Connect(address))
            {
                return false;
            }

            bool result = true;

            try
            {
                telnet.Login();
                version = telnet.WaitCommand("hexdump -s 7 -n 1 -e '\"%d\"' /sys/bus/i2c/devices/0-0054/eeprom");
            }
            catch (Exception e)
            {
                if (Verbose)
                {
                    Console.WriteLine(e);
                }
                result = false;
            }

            telnet.Exit();
            return result;
        }
    }
}
