﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.Bridge
{
    public class BridgeControl : IDisposable
    {
        public enum LED_ID { POWER, SD, GC };
        public enum LED_MODE { OFF, ON, BLINK, NORMAL };

        const int GPIO_SD_GC_ALTERNATE = 26;

        const int GPIO_POWER_LED_BLINK = 23;
        const int GPIO_POWER_LED_OUT   = 5;
        const int GPIO_POWER_LED_EN    = 22;

        const int GPIO_SD_LED_BLINK    = 25;
        const int GPIO_SD_LED_OUT      = 11;
        const int GPIO_SD_LED_EN       = 10;

        const int GPIO_GC_LED_BLINK    = 24;
        const int GPIO_GC_LED_OUT      = 8;
        const int GPIO_GC_LED_EN       = 7;

        const string initPowerLed   = "ngpio_test -p23 -d1; ngpio_test -p22 -d1; ngpio_test -p5  -d1; ";
        const string initSdLed      = "ngpio_test -p25 -d1; ngpio_test -p11 -d1; ngpio_test -p10 -d1; ";
        const string initGcLed      = "ngpio_test -p24 -d1; ngpio_test -p8  -d1; ngpio_test -p7  -d1; ";

        const string onPowerLed     = "ngpio_test -p23 -w0; ngpio_test -p22 -w1; ngpio_test -p5  -w1; ";
        const string onSdLed        = "ngpio_test -p25 -w0; ngpio_test -p11 -w1; ngpio_test -p10 -w0; ";
        const string onGcLed        = "ngpio_test -p24 -w0; ngpio_test -p8  -w1; ngpio_test -p7  -w0; ";

        const string offPowerLed    = "ngpio_test -p23 -w0; ngpio_test -p22 -w1; ngpio_test -p5  -w0; ";
        const string offSdLed       = "ngpio_test -p25 -w0; ngpio_test -p11 -w1; ngpio_test -p10 -w1; ";
        const string offGcLed       = "ngpio_test -p24 -w0; ngpio_test -p8  -w1; ngpio_test -p7  -w11; ";

        const string blinkPowerLed  = "ngpio_test -p23 -w1; ";
        const string blinkSdLed     = "ngpio_test -p25 -w1; ";
        const string blinkGcdLed    = "ngpio_test -p24 -w1; ";

        const string normalPowerLed = "ngpio_test -p23 -w0; ngpio_test -p22 -w0; ";
        const string normalSdLed    = "ngpio_test -p25 -w0; ngpio_test -p11 -w0; ";
        const string normalGcLed    = "ngpio_test -p24 -w0; ngpio_test -p8  -w0; ";

        const string bootForceRecovery  = "hbtool button -f";

        Telnet telnet = null;

        public BridgeControl()
        {
        }

        public BridgeControl(string address)
        {
            if(!Open(address))
            {
                throw new Exception("Failed to login hostbridge");
            }
        }

        public void Dispose()
        {
            Close();
        }

        public bool Open(string address)
        {
            telnet = new Telnet();
            if (!telnet.Connect(address))
            {
                return false;
            }

            if (!telnet.Login())
            {
                return false;
            }

            try
            {
                telnet.Send(initPowerLed + initSdLed + initGcLed);
                telnet.Wait("# ");
            }
            catch (Exception)
            {
                telnet.Exit();
                telnet = null;
                return false;
            }

            return true;
        }

        public bool LED(LED_ID led, LED_MODE mode)
        {
            bool result = true;

            try
            {
                switch (led)
                {
                    case LED_ID.POWER:
                        switch (mode)
                        {
                            case LED_MODE.ON:
                                telnet.Send(onPowerLed);
                                break;

                            case LED_MODE.OFF:
                                telnet.Send(offPowerLed);
                                break;

                            case LED_MODE.BLINK:
                                telnet.Send(blinkPowerLed);
                                break;

                            case LED_MODE.NORMAL:
                                telnet.Send(normalPowerLed);
                                break;
                        }
                        break;

                    case LED_ID.SD:
                        switch (mode)
                        {
                            case LED_MODE.ON:
                                telnet.Send(onSdLed);
                                break;

                            case LED_MODE.OFF:
                                telnet.Send(offSdLed);
                                break;

                            case LED_MODE.BLINK:
                                telnet.Send(blinkSdLed);
                                break;

                            case LED_MODE.NORMAL:
                                telnet.Send(normalSdLed);
                                break;
                        }
                        break;

                    case LED_ID.GC:
                        switch (mode)
                        {
                            case LED_MODE.ON:
                                telnet.Send(onGcLed);
                                break;

                            case LED_MODE.OFF:
                                telnet.Send(offGcLed);
                                break;

                            case LED_MODE.BLINK:
                                telnet.Send(blinkGcdLed);
                                break;

                            case LED_MODE.NORMAL:
                                telnet.Send(normalGcLed);
                                break;
                        }
                        break;

                    default:
                        return false;
                }
            }
            catch (Exception e)
            {
                Console.WriteLine(e);
                result = false;
            }

            return result;
        }

        public void ForceRecovery()
        {
            telnet.Send(bootForceRecovery);
            telnet.Wait("# ");
        }

        public void Close()
        {
            if (telnet != null)
            {
                System.Threading.Thread.Sleep(10);
                telnet.Exit();
                telnet = null;
            }
        }
    }
}
