﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nn.Adl.Syntax;
using Nn.Adl.Parsing;
using Nn.ResultTool.Syntax;

namespace Nn.ResultTool.Parsing
{
    using TokenKind = Nn.ResultTool.Parsing.Autogen.Token;

    internal class ResultParser
    {
        private static readonly DefaultTokenReader s_TokenReader = new DefaultTokenReader(@"{}[]:=();,<>$", new[] { "[[", "]]" });
        private static readonly TokenKindGetter s_TokenKindGetter = new TokenKindGetter();

        public static Document Parse(string text, string initialPath)
        {
            var tokenStream = new DefaultTokenStream<TokenValue>(text, s_TokenReader);
            return new Parser().Parse(tokenStream, initialPath, s_TokenKindGetter);
        }

        private class Parser
            : Nn.ResultTool.Parsing.Autogen.Parser, IParser<TokenKind>
        {
            public Parser() : base(new SemanticAction()) { }

            public dynamic GetParserStack()
            {
                return this.stack;
            }
        }

        private class TokenKindGetter : ITokenKindGetter<TokenKind>
        {
            public TokenKind? GetSymbolTokenKind(TokenValue tokenValue)
            {
                switch (tokenValue.Text)
                {
                    case "{": return TokenKind.token_BRACE_LEFT;
                    case "}": return TokenKind.token_BRACE_RIGHT;
                    case "[": return TokenKind.token_BRACKET_LEFT;
                    case "]": return TokenKind.token_BRACKET_RIGHT;
                    case ":": return TokenKind.token_COLON;
                    case ",": return TokenKind.token_COMMA;
                    case "$": return TokenKind.token_DOLLAR;
                    //case ".": return TokenKind.token_DOT;
                    case "=": return TokenKind.token_EQUAL;
                    case ">": return TokenKind.token_GREATER;
                    //case "?": return TokenKind.token_HATENA;
                    case "<": return TokenKind.token_LESS;
                    case "(": return TokenKind.token_PAREN_LEFT;
                    case ")": return TokenKind.token_PAREN_RIGHT;
                    //case "+": return TokenKind.token_PLUS;
                    case ";": return TokenKind.token_SEMICOLON;
                    case "[[": return TokenKind.token_DOUBLE_BRACE_LEFT;
                    case "]]": return TokenKind.token_DOUBLE_BRACE_RIGHT;
                    default: return null;
                }
            }

            public TokenKind? GetKeywordTokenKind(TokenValue tokenValue)
            {
                switch (tokenValue.Text)
                {
                    case "false": return TokenKind.token_FALSE;
                    case "true": return TokenKind.token_TRUE;
                    default: return null;
                }
            }

            public TokenKind GetIntegerTokenKind()
            {
                return TokenKind.token_INTEGER;
            }

            public TokenKind GetIdentifierTokenKind()
            {
                return TokenKind.token_IDENTIFIER;
            }

            public TokenKind GetStringTokenKind()
            {
                return TokenKind.token_STRING;
            }

            public TokenKind GetEofTokenKind()
            {
                return TokenKind.token_eof;
            }
        }

        private class SemanticAction
            : Nn.ResultTool.Parsing.Autogen.SubSemanticActionsAdapter, Nn.ResultTool.Parsing.Autogen.ISemanticAction
        {
            public SemanticAction()
            {
                this.ErrorHandlingSemanticAction = new ErrorHandlingSemanticAction();
                this.AdlCommonSubSemanticAction = new AdlCommonSubSemanticAction();
                this.ResultSubSemanticAction = new ResultSubSemanticAction();
            }
        }

        private class ErrorHandlingSemanticAction
            : Nn.Adl.Parsing.ErrorHandlingSemanticAction, Nn.ResultTool.Parsing.Autogen.IErrorHandlingSemanticAction
        {
        }

        private class AdlCommonSubSemanticAction
            : Nn.Adl.Parsing.AdlCommonSubSemanticAction, Nn.ResultTool.Parsing.Autogen.IAdlCommonSubSemanticAction
        {
        }

        private class ResultSubSemanticAction : Nn.ResultTool.Parsing.Autogen.IResultSubSemanticAction
        {
            public IEnumerable<Statement> MakeChildren(int pos, IEnumerable<Statement> list)
            {
                return list;
            }

            public Expression MakeExpression(int pos, Literal<long> value)
            {
                return new LiteralExpression
                {
                    Literal = value,
                };
            }

            public Expression MakeStringLiteral(int pos, Literal<string> value)
            {
                return new LiteralExpression
                {
                    Literal = value,
                };
            }

            public Expression MakeStringLiteral(int pos, Symbol symbol, Literal<string> value)
            {
                return new SymboledLiteralExpression
                {
                    Symbol = symbol,
                    Literal = value,
                };
            }

            public Expression MakeResultExpression(int pos, Literal<long> begin)
            {
                return new LiteralExpression
                {
                    Literal = Literal.Create(new ResultRangeValue { Begin = (int)begin.Value, End = (int)begin.Value + 1 }),
                };
            }

            public Expression MakeResultExpression(int pos, Literal<long> begin, Literal<long> end)
            {
                return new LiteralExpression
                {
                    Literal = Literal.Create(new ResultRangeValue { Begin = (int)begin.Value, End = (int)end.Value }),
                };
            }

            public Modification MakeDirectModificationExpression(int pos, Literal<string> description)
            {
                return new NamelessModification
                {
                    ValueExpression = new LiteralExpression { Literal = description },
                };
            }

            private Expression MakeBool(int pos, Keyword value, bool b)
            {
                return new LiteralExpression
                {
                    Literal = Literal.Create(b),
                };
            }

            public Expression MakeFalse(int pos, Keyword value)
            {
                return MakeBool(pos, value, false);
            }

            public Expression MakeTrue(int pos, Keyword value)
            {
                return MakeBool(pos, value, true);
            }
        }
    }
}
