﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Globalization;
using System.IO;

namespace Nn.Adl.Parsing
{
    internal static class StringUtility
    {
        /// <summary>
        /// 指定された位置から後方で最も近い行頭の位置を返します。
        /// </summary>
        /// <param name="s">対象の文字列</param>
        /// <param name="start">探索を開始する位置</param>
        /// <returns>改行が見つかった場合は改行と次の行の先頭の間を指すの位置を返します。
        /// 見つからなかった場合は負の値を返します。</returns>
        public static int IndexOfLineHead(this string s, int start)
        {
            var found = s.IndexOfAny(new char[] { '\n', '\r' }, start);

            if (found < 0)
            {
                return found;
            }
            if (s[found] == '\r'
                && found + 1 < s.Length
                && s[found + 1] == '\n')
            {
                return found + 2;
            }

            return found + 1;
        }

        /// <summary>
        /// 指定された位置から後方で最も近い行末の位置を返します。
        /// </summary>
        /// <param name="s">対象の文字列</param>
        /// <param name="start">探索を開始する位置</param>
        /// <returns>改行が見つかった場合は改行の直前の位置を返します。
        /// 見つからなかった場合は文字列の末尾の位置を返します。</returns>
        public static int IndexOfLineEnd(this string s, int start)
        {
            var found = s.IndexOfAny(new char[] { '\n', '\r' }, start);
            return (found >= 0) ? found : s.Length - 1;
        }

        /// <summary>
        /// 指定された位置から前方で最も近い行頭位置を返します。
        /// </summary>
        /// <param name="s">対象の文字列</param>
        /// <param name="start">探索を開始する位置</param>
        /// <returns>改行が見つかった場合は改行と次の行の先頭の間を指す位置を返します。
        /// 見つからなかった場合は 0 を返します</returns>
        public static int LastIndexOfLineHead(this string s, int start)
        {
            if (start <= 0)
            {
                return 0;
            }

            // start の位置に改行があるとそれがヒットしてしまうため
            // -1 する
            --start;

            var found = s.LastIndexOfAny(new char[] { '\n', '\r' }, start);
            return (found < 0) ? 0 : found + 1;
        }

        /// <summary>
        /// 指定された位置から前方で最も近い行末の位置を返します。
        /// </summary>
        /// <param name="s">対象の文字列</param>
        /// <param name="start">探索を開始する位置</param>
        /// <returns>改行が見つかった場合は改行の直前の位置を返します。
        /// 見つからなかった場合は負の値を返します</returns>
        public static int LastIndexOfLineEnd(this string s, int start)
        {
            // start の位置に改行があるとそれがヒットしてしまうため
            // -1 する
            if (start > 0)
            {
                --start;
            }
            var found = s.LastIndexOfAny(new char[] { '\n', '\r' }, start);
            if (found < 0)
            {
                return found;
            }
            if (s[found] == '\n'
                && found - 1 >= 0
                && s[found - 1] == '\r')
            {
                return found - 1;
            }
            return found;
        }

        public class TextPoint
        {
            public int Lines { get; set; }
            public int Chars { get; set; }
        }

        public static TextPoint GetTextPoint(string s, int pos)
        {
            int lineNo = 1;
            int prevPos = 0;
            int nextPos = 0;

            while (true)
            {
                nextPos = s.IndexOfLineHead(prevPos);

                if (nextPos < 0 || nextPos > pos)
                {
                    break;
                }
                prevPos = nextPos;
                lineNo++;
            }

            int posInLine = pos - prevPos;

            return new TextPoint() { Lines = lineNo, Chars = posInLine };
        }

        public static string FormatTextPoint(string path, TextPoint tp, int tabOffset)
        {
            if (tp.Chars >= 0 && tp.Lines >= 0)
            {
                return string.Format("line {0,3} pos {1,3} ({3,3}) file {4}| ( {5}({0},{2}) )",
                    tp.Lines, tp.Chars, 1 + tp.Chars, 1 + tp.Chars + tabOffset,
                    Path.GetFileName(path),
                    Path.GetFullPath(path));
            }
            else
            {
                return string.Format(" {0}", path);
            }
        }
    }
}
