﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeSvcVeneer
{
    internal class SvcHandlerTableGenerator : SourceGenerator
    {
        private string m_handlerHeaderPath;
        private string m_driverHeaderPath;
        private string m_selfPath;

        public SvcHandlerTableGenerator(CodeGenNames p) : base(p) { }

        public void Generate(
            SvcSet ss,
            Dictionary<string, LayoutConversion> conv,
            string templatePath,
            string path,
            string handlerHeaderPath,
            string driverHeaderPath)
        {
            this.m_driverHeaderPath = driverHeaderPath;
            this.m_handlerHeaderPath = handlerHeaderPath;
            this.m_selfPath = path;

            this.Generate(ss, null, null, conv, templatePath, path);
        }

        protected override string Generate(
            SvcSet ss,
            Dictionary<string, AbiLayout> abi,
            Dictionary<string, SvcLayout> svc,
            Dictionary<string, LayoutConversion> conv)
        {
            var fnMap = ss.Operations.ToDictionary(x => x.FunctionNumber, x => x);

            var driverPrefix = string.Format("{0}::{1}::{2}",
                            CodeGenNames.KernelCommonNamespace,
                            Params.DriverClassName,
                            CodeGenNames.DriverNamePrefix);
            var handlerPrefix = string.Format("{0}::{1}",
                            CodeGenNames.KernelCommonNamespace,
                            CodeGenNames.HandlerNamePrefix);

            var prefixLength = Math.Max(driverPrefix.Length, handlerPrefix.Length);
            var lineFormat = string.Format("{0}{1}{2}",
                "    /* {0,3} */  reinterpret_cast<void*>({1,",
                prefixLength,
                "}{2}{3}),\r\n");

            var selfDir = Path.GetDirectoryName(this.m_selfPath);
            var handlerHeaderRelativePath = Util.MakeRelativePath(selfDir, this.m_handlerHeaderPath);
            var driverHeaderRelativePath = Util.MakeRelativePath(selfDir, this.m_driverHeaderPath);

            var sb = new StringBuilder();
            sb.Append(SourceGenerator.MakeIncludeLine(handlerHeaderRelativePath));
            sb.Append(SourceGenerator.MakeIncludeLine(driverHeaderRelativePath));
            sb.AppendLine();
            sb.AppendFormat("extern \"C\" void* const {0}[128] =\r\n", Params.HandlerTableName);
            sb.AppendLine("{");

            for (int i = 0; i < 128; ++i)
            {
                var f = Util.GetValueOrDefault(fnMap, i);

                if (f == null)
                {
                    sb.AppendFormat("    /* {0,3} */  NULL,\r\n", i);
                }
                else
                {
                    string prefix;
                    string postfix;
                    if (conv[f.Name].IsRequireConversion)
                    {
                        prefix = driverPrefix;
                        postfix = string.Empty;
                    }
                    else
                    {
                        prefix = handlerPrefix;
                        postfix = Params.HandlerNamePostfix;
                    }
                    sb.AppendFormat(lineFormat, i, prefix, f.Name, postfix);
                }
            }

            sb.AppendLine("};");
            return sb.ToString();
        }
    }
}
