﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeSvcVeneer
{
    /// <summary>
    /// 一つの関数のあるタイミングの引数レイアウトを表します。
    /// </summary>
    internal class Layout
    {
        private const uint StackLinearPositionOffset = 10000;

        public enum StorageType
        {
            Register,
            Stack,
        }

        /// <summary>
        /// 引数の格納場所を表します。
        /// </summary>
        public class Position : IComparable<Position>
        {
            /// <summary>
            /// ストレージの種別を取得します。
            /// </summary>
            public StorageType Storage { get; private set; }
            /// <summary>
            /// ストレージ内の格納場所の番号を取得します。
            /// ストレージがレジスタならレジスタ番号、
            /// スタックならスタックのトップからのワード数です。
            /// </summary>
            public int Index { get; private set; }

            public Position(StorageType s, int i)
            {
                Storage = s;
                Index = i;
            }
            public bool IsSame(Position p)
            {
                return (this.Storage == p.Storage)
                    && (this.Index == p.Index);
            }

            #region IComparable<Position> メンバー

            public int CompareTo(Position other)
            {
                if (this.Storage != other.Storage)
                {
                    return (this.Storage == StorageType.Register) ? -1 : 1;
                }

                return this.Index - other.Index;
            }

            #endregion
        }

        /// <summary>
        /// 仮引数のレイアウト情報を表します。
        /// </summary>
        public class StorageEntry
        {
            /// <summary>
            /// 仮引数の名前を取得します。
            /// </summary>
            public string VariableName { get; private set; }
            /// <summary>
            /// 引数の値が格納されている場所のリストを取得します。
            /// リスト内の順番通りに値が分割されて格納されます。
            /// </summary>
            public List<Position> Positions { get; private set; }
            /// <summary>
            /// 引数が参照渡しされるか否かを取得します。
            /// </summary>
            public bool IsReference { get; private set; }

            /// <summary>
            /// 引数に対応するレイアウト上でのサイズです。
            /// </summary>
            public int VariableTransferSize { get; private set; }
            /// <summary>
            /// 引数が指すオブジェクトのサイズです。
            /// </summary>
            public int VariableDataSize { get; private set; }

            public StorageEntry(string name, Position p, int transferSize, int dataSize, bool isReference)
            {
                VariableName = name;
                Positions = new List<Position>() { p };
                IsReference = isReference;
                VariableTransferSize = transferSize;
                VariableDataSize = dataSize;
            }
            public void AddIndex(Position p) { Positions.Add(p); }
        }

        private Dictionary<string, StorageEntry> m_paramPosition;
        private Dictionary<uint, StorageEntry> m_positionParam;

        public int StorageSize { get; private set; }
        public int RegisterCount { get; private set; }
        public int PointerSize { get; private set; }
        public int RegisterSize { get; private set; }

        public IEnumerable<StorageEntry> Params
        {
            get
            {
                return m_paramPosition.Values
                        .OrderBy(x => SerializePosition(x.Positions[0]));
            }
        }
        public IEnumerable<Tuple<Position, StorageEntry>> StorageParams
        {
            get
            {
                return m_positionParam.OrderBy(x => x.Key)
                    .Select(x => Tuple.Create(DeserializePosition(x.Key), x.Value));
            }
        }

        public Layout(int s, int r, int p, int rs)
        {
            StorageSize = s;
            RegisterCount = r;
            PointerSize = p;
            RegisterSize = p;

            m_paramPosition = new Dictionary<string, StorageEntry>();
            m_positionParam = new Dictionary<uint, StorageEntry>();
        }

        public void AddOne(string name, StorageType s, int index, int transferSize, int dataSize,  bool isReference)
        {
            var p = new Position(s, index);
            var lp = SerializePosition(p);

            StorageEntry se;
            if (m_paramPosition.TryGetValue(name, out se))
            {
                se.AddIndex(p);
            }
            else
            {
                se = new StorageEntry(name, p, transferSize, dataSize, isReference);
                m_paramPosition.Add(name, se);
            }

            m_positionParam.Add(lp, se);
        }
        public int Add(string name, StorageType s, int index, int transferSize, int dataSize, bool isPointer = false)
        {
            int usedRegs = 0;
            while (usedRegs * StorageSize < transferSize)
            {
                AddOne(name, s, index++, transferSize, dataSize, isPointer);
                usedRegs++;
            }
            return usedRegs;
        }

        /// <summary>
        /// 対象のレジスタが空いているかを判定します。
        /// </summary>
        /// <param name="index">判定対象のレジスタ番号</param>
        /// <returns>index 番のレジスタが未使用か否か</returns>
        public bool IsRegisterAvailable(int index)
        {
            var p = new Position(StorageType.Register, index);
            var lp = SerializePosition(p);
            return !m_positionParam.ContainsKey(lp);
        }
        /// <summary>
        /// 対象のレジスタに格納されている仮引数の名前を取得します。
        /// </summary>
        /// <param name="index">対象とするレジスタ</param>
        /// <returns>仮引数の名前</returns>
        public string GetRegisterParam(int index)
        {
            var p = new Position(StorageType.Register, index);
            StorageEntry se;
            if (m_positionParam.TryGetValue(SerializePosition(p), out se))
            {
                return se.VariableName;
            }
            return null;
        }
        /// <summary>
        /// 対象の仮引数の値が格納されている場所の一つを取得します
        /// </summary>
        /// <param name="name">対象とする仮引数の名前</param>
        /// <returns>仮引数の値が格納されている場所の一つ</returns>
        public Position GetPosition(string name)
        {
            return m_paramPosition[name].Positions[0];
        }
        /// <summary>
        /// 対象の仮引数の値が格納されている場所を取得します。
        /// 返り値の順番通りに値が分割されて格納されます。
        /// </summary>
        /// <param name="name">対象とする仮引数の名前</param>
        /// <returns>仮引数の値が格納されている場所のリスト</returns>
        public Position[] GetPositions(string name)
        {
            var se = Util.GetValueOrDefault(m_paramPosition, name);
            if (se == null)
            {
                return null;
            }
            return se.Positions.ToArray();
        }
        /// <summary>
        /// 対象の仮引数のレイアウト情報を取得します。
        /// 指定された名前の仮引数がない場合は null を返します。
        /// </summary>
        /// <param name="name">対象とする仮引数の名前</param>
        /// <returns>仮引数のレイアウト情報</returns>
        public StorageEntry GetEntryOrNull(string name)
        {
            return Util.GetValueOrDefault(m_paramPosition, name);
        }
        /// <summary>
        /// 対象の仮引数のレイアウト情報を取得します。
        /// 指定された名前の仮引数がない場合は例外を投げます。
        /// </summary>
        /// <param name="name">対象とする仮引数の名前</param>
        /// <returns>仮引数のレイアウト情報</returns>
        public StorageEntry GetEntry(string name)
        {
            return m_paramPosition[name];
        }

        private Position DeserializePosition(uint lp)
        {
            if (lp < StackLinearPositionOffset)
            {
                return new Position(StorageType.Register, (int)lp);
            }
            else
            {
                return new Position(StorageType.Stack, (int)(lp - StackLinearPositionOffset));
            }
        }
        private uint SerializePosition(Position p)
        {
            if (p.Storage == StorageType.Register)
            {
                return (uint)p.Index;
            }
            else
            {
                return (uint)(StackLinearPositionOffset + p.Index);
            }
        }
    }
}
