﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace MakeSvcVeneer
{
    internal class SvcLayout
    {
        /// <summary>
        /// システムコール呼び出し前の引数レイアウト情報を取得します。
        /// </summary>
        public Layout In { get; private set; }
        /// <summary>
        /// システムコール呼び出し後の引数レイアウト情報を取得します。
        /// </summary>
        public Layout Out { get; private set; }

        public CodeGenParams CodeGenParams { get; private set; }

        private SvcLayout(CodeGenParams ap)
        {
            this.CodeGenParams = ap;
            In = new Layout(ap.StorageSize, ap.RegisterCount, ap.PointerSize, ap.RegisterSize);
            Out = new Layout(ap.StorageSize, ap.RegisterCount, ap.PointerSize, ap.RegisterSize);
        }

        public static Dictionary<string, SvcLayout> Arrange(
            Operation[] operations,
            Dictionary<string, AbiLayout> abiLayouts,
            CodeGenParams ap)
        {
            var map = new Dictionary<string, SvcLayout>();

            foreach (var op in operations)
            {
                try
                {
                    map.Add(op.Name, Arrange(op, abiLayouts[op.Name], ap));
                }
                catch (ErrorException ee)
                {
                    var sb = new StringBuilder();
                    sb.AppendFormat("operation: {0} (id={1})\n", op.Name, op.FunctionNumber);
                    sb.AppendFormat("return:    {0} ({1} byte)\n", op.ReturnType.Name, op.ReturnType.Size);
                    sb.AppendFormat("params:\n");
                    sb.AppendFormat("  name              size array  dir     type\n");
                    sb.AppendFormat("  ----------------------------------------------------\n");
                    foreach (var p in op.Parameters)
                    {
                        var size = p.IsPointerForced ? ap.PointerSize.ToString() : p.GetSizeText(ap.PointerSize, ap.RegisterSize);
                        sb.AppendFormat("  {0,-16}   {2,2}  {3,-5}  {4,-6}  {1}\n",
                            p.Name,
                            p.Type.Name,
                            size,
                            p.IsArray,
                            p.Direction);
                    }

                    throw new ErrorException(ee.Message, sb.ToString());
                }
            }

            return map;
        }

        private class ParamInfo
        {
            public bool IsPointer { get; private set; }
            public int Size { get; private set; }
            public OperationParameter Parameter { get; private set; }

            public ParamInfo(OperationParameter op, int size, bool p)
            {
                IsPointer = p;
                Size = size;
                Parameter = op;
            }
        }
        private class RegisterUsageManager
        {
            private bool[] m_registerUsedMap;

            public RegisterUsageManager(int registerCount)
            {
                m_registerUsedMap = new bool[registerCount];
            }
            public void AllocateAt(int index)
            {
                m_registerUsedMap[index] = true;
            }
            public int AllocateAuto(string name)
            {
                var index = m_registerUsedMap.FindIndex(x => !x);
                if (index < 0)
                {
                    throw new ErrorException(
                        string.Format("レジスタに載りきりません。({0})", name));
                }

                m_registerUsedMap[index] = true;
                return index;
            }
        }

        private static void AddDirectRegister(
            Layout svc,
            RegisterUsageManager usage,
            IEnumerable<Layout.StorageEntry> abi)
        {
            foreach (var se in abi)
            {
                foreach (var pos in se.Positions)
                {
                    if (pos.Storage == Layout.StorageType.Register)
                    {
                        usage.AllocateAt(pos.Index);
                        svc.AddOne(se.VariableName, pos.Storage,
                            pos.Index, se.VariableTransferSize, se.VariableDataSize, se.IsReference);
                    }
                }
            }
        }
        private static void AddDirectStacks(
            Layout svc,
            RegisterUsageManager usage,
            IEnumerable<Layout.StorageEntry> abi,
            string opName)
        {
            foreach (var se in abi)
            {
                foreach (var pos in se.Positions)
                {
                    if (pos.Storage == Layout.StorageType.Stack)
                    {
                        var index = usage.AllocateAuto(opName);
                        svc.AddOne(se.VariableName, Layout.StorageType.Register,
                            index, se.VariableTransferSize, se.VariableDataSize, se.IsReference);
                    }
                }
            }
        }
        private static void AddNewAssigns(
            Layout svc,
            RegisterUsageManager usage,
            IEnumerable<OperationParameter> ops,
            string opName)
        {
            foreach (var p in ops)
            {
                // AbiLayout の時点で撥ねられているはず
                Debug.Assert(!p.IsUnknownSize, "assertion failed");

                var dataSize = p.GetSize(svc.PointerSize, svc.RegisterSize);
                for (int offset = 0; offset < dataSize; offset += svc.StorageSize)
                {
                    var index = usage.AllocateAuto(opName);
                    svc.AddOne(p.Name, Layout.StorageType.Register, index, dataSize, dataSize, false);
                }
            }
        }

        private static void ArrangeIn(Layout slin, Operation op, Layout alin)
        {
            var inRegisterUsage = new RegisterUsageManager(slin.RegisterCount);

            var ins = op.Parameters
                        .Where(x => x.Direction == OperationParameter.DirectionType.In);

            var inValues = ins.Select(x => alin.GetEntry(x.Name))
                              .Where(x => !x.IsReference);
            var inRefs   = ins.Where(x => alin.GetEntry(x.Name).IsReference);

            var inptrs = op.Parameters
                            .Where(x => x.Direction == OperationParameter.DirectionType.InPtr)
                            .Select(x => alin.GetEntry(x.Name));
            var outptrs = op.Parameters
                            .Where(x => x.Direction == OperationParameter.DirectionType.OutPtr)
                            .Select(x => alin.GetEntry(x.Name));
            var directTransfers = inValues.Concat(inptrs).Concat(outptrs)
                .OrderBy(x => x.Positions[0]);

            // in であり、値渡しであり、レジスタ渡しのものは ABI レイアウトのまま維持する
            // inptr であり、レジスタ渡しのものは ABI レイアウトのまま維持する
            // outptr であり、レジスタ渡しのものは ABI レイアウトのまま維持する
            AddDirectRegister(slin, inRegisterUsage, directTransfers);

            // in であり、値渡しであり、スタック渡しのものはレジスタに載せる
            // inptr であり、スタック渡しのものはレジスタに載せる
            // outptr であり、スタック渡しのものはレジスタに載せる
            AddDirectStacks(slin, inRegisterUsage, directTransfers, op.Name);

            // in であり、参照渡しのものは値渡しに展開
            AddNewAssigns(slin, inRegisterUsage, inRefs, op.Name);
        }
        private static void ArrangeOut(Layout slout, Operation op, Layout alout)
        {
            var outRegisterUsage = new RegisterUsageManager(slout.RegisterCount);

            // ABI の out は ABI レイアウトのまま維持する
            // return だけのはず
            Debug.Assert(alout.Params.All(x => x.VariableName == "return"), "assertion failed");
            AddDirectRegister(slout, outRegisterUsage, alout.Params);

            // out であり、値渡しのものは out に配置
            // out であり、参照渡しのものは out で値渡しに展開
            var outs = op.Parameters
                         .Where(x => x.Direction == OperationParameter.DirectionType.Out);
            AddNewAssigns(slout, outRegisterUsage, outs, op.Name);
        }
        private static SvcLayout Arrange(Operation op, AbiLayout al, CodeGenParams ap)
        {
            // op --- abi ---------
            //
            // in     値   レジスタ  ABI レイアウトのまま維持
            //             スタック  Move
            //        参照 レジスタ  Scatter
            //             スタック  IndirectScatter
            // inptr  参照 レジスタ  ABI レイアウトのまま維持
            //             スタック  Move
            // out    値   レジスタ  out へ
            //             スタック  out へ
            //        参照 レジスタ  out へ
            //             スタック  out へ
            // outptr 参照 レジスタ  ABI レイアウトのまま維持
            //             スタック  Move

            SvcLayout sl = new SvcLayout(ap);
            ArrangeIn (sl.In,  op, al.In);
            ArrangeOut(sl.Out, op, al.Out);
            return sl;
        }
    }
}
