﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.InteropServices;
using MakeNro.Elf;

namespace MakeNro
{
    /// <summary>
    /// NRO オブジェクトのヘッダ情報
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    internal struct NroHeader
    {
        public const int SignatureSize      = 0x4;
        public const int ModuleIdSize       = 0x20;
        public const int ReservedSize       = 0x04;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = SignatureSize)]
        public byte[] Signature;
        public uint Version;
        public uint Size;
        public uint Flags;
        public uint TextMemoryOffset;
        public uint TextSize;
        public uint RoMemoryOffset;
        public uint RoSize;
        public uint DataMemoryOffset;
        public uint DataSize;
        public uint BssSize;
        public uint Reserved0x3C;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = ModuleIdSize)]
        public byte[] ModuleId;
        public uint DsoHandleOffset;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = ReservedSize)]
        public byte[] Reserved;
        public uint EmbededOffset;
        public uint EmbededSize;
        public uint DynStrOffset;
        public uint DynStrSize;
        public uint DynSymOffset;
        public uint DynSymSize;
    }

    /// <summary>
    /// NRO  ファイルを表現するクラス
    /// </summary>
    internal class NroFile
    {
        private NroHeader header;
        private uint headerSize;
        private byte[] textBinary;
        private byte[] roBinary;
        private byte[] dataBinary;
        private uint bssMemoryOffset;
        private const int NroHeaderStartOffset = 0x10;

        public NroFile()
        {
            header = new NroHeader();
            header.Signature = new byte[NroHeader.SignatureSize];
            header.Signature[0] = (byte)'N';
            header.Signature[1] = (byte)'R';
            header.Signature[2] = (byte)'O';
            header.Signature[3] = (byte)'0';
            header.ModuleId = new byte[NroHeader.ModuleIdSize];
            header.Reserved = new byte[NroHeader.ReservedSize];
            headerSize = (uint)Marshal.SizeOf(typeof(NroHeader));

            // TORIAEZU
            header.Flags = 0;
        }

        public void SetModuleId(byte[] moduleId)
        {
            moduleId.CopyTo(header.ModuleId, 0);
        }

        /// <summary>
        /// api_infoセクションの情報を設定します
        /// </summary>
        /// <param name="info">api_infoセクションのオフセット</param>
        /// <param name="info">api_infoセクションのサイズ</param>
        public void SetApiInfo(ulong offset, ulong size)
        {
            header.EmbededOffset = (uint)offset;
            header.EmbededSize = (uint)size;
        }

        /// <summary>
        /// dynstrセクションの情報を設定します
        /// </summary>
        /// <param name="info">dynstrセクションのオフセット</param>
        /// <param name="info">dynstrセクションのサイズ</param>
        public void SetDynStrInfo(ulong offset, ulong size)
        {
            header.DynStrOffset = (uint)offset;
            header.DynStrSize = (uint)size;
        }

        /// <summary>
        /// dynsymセクションの情報を設定します
        /// </summary>
        /// <param name="info">dynsymセクションのオフセット</param>
        /// <param name="info">dynsymセクションのサイズ</param>
        public void SetDynSymInfo(ulong offset, ulong size)
        {
            header.DynSymOffset = (uint)offset;
            header.DynSymSize = (uint)size;
        }

        /// <summary>
        /// __dso_handle 取得用関数の情報を設定します
        /// </summary>
        /// <param name="info">GetDsoHandle関数のオフセット</param>
        public void SetDsoHandleOffset(ulong offset)
        {
            header.DsoHandleOffset = (uint)offset;
        }

        /// <summary>
        /// Text セグメントの情報を設定します
        /// </summary>
        /// <param name="info">セグメントの情報</param>
        public void SetTextSegment(ElfSegmentInfo info)
        {
            if (info == null)
            {
                return;
            }

            if (info.VirtualAddress != 0)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidTextAddress, info.VirtualAddress));
            }

            header.TextMemoryOffset = (uint)info.VirtualAddress;
            header.TextSize = (uint)info.MemorySize;
            textBinary = info.GetContents();
        }

        /// <summary>
        /// Ro セグメントの情報を設定します
        /// </summary>
        /// <param name="info">セグメントの情報</param>
        public void SetRoSegment(ElfSegmentInfo info)
        {
            if (info == null)
            {
                return;
            }

            if ((info.VirtualAddress & 0xFFF) > 0)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidSegmentAlign, "RO", info.VirtualAddress));
            }

            header.RoMemoryOffset = (uint)info.VirtualAddress;
            header.RoSize = (uint)info.MemorySize;
            roBinary = info.GetContents();
        }

        /// <summary>
        /// Data セグメントの情報を設定します
        /// </summary>
        /// <param name="info">セグメントの情報</param>
        public void SetDataSegment(ElfSegmentInfo info)
        {
            if (info == null)
            {
                return;
            }

            if ((info.VirtualAddress & 0xFFF) > 0)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidSegmentAlign, "Data", info.VirtualAddress));
            }

            header.DataMemoryOffset = (uint)info.VirtualAddress;
            header.DataSize = (uint)info.MemorySize;
            dataBinary = info.GetContents();
        }

        /// <summary>
        /// Bss セグメントの情報を設定します
        /// </summary>
        /// <param name="info">セグメントの情報</param>
        public void SetBssSegment(ElfSegmentInfo info)
        {
            if (info == null)
            {
                return;
            }

            if ((info.VirtualAddress & 0xF) > 0)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidBssSegmentAlign, info.VirtualAddress));
            }

            header.BssSize = (uint)info.MemorySize;
            bssMemoryOffset = (uint)info.VirtualAddress;
        }

        /// <summary>
        /// 設定された情報から位置情報を調整します。
        /// </summary>
        public void CalcPosition()
        {
            // ヘッダに埋め込む各種サイズを実機側で扱いやすい値に整理する
            header.TextSize = AlignUp((uint)header.TextSize, 0x1000);
            header.RoSize = AlignUp((uint)header.RoSize, 0x1000);

            if (header.TextMemoryOffset + header.TextSize != header.RoMemoryOffset)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidRoAddress, header.TextMemoryOffset, header.TextSize, header.RoMemoryOffset));
            }
            if (header.RoMemoryOffset + header.RoSize != header.DataMemoryOffset)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidDataAddress, header.RoMemoryOffset, header.RoSize, header.DataMemoryOffset));
            }
            if (header.BssSize > 0)
            {
                // Data サイズが 0 だったときは、配置オフセットに BSS の先頭アドレスを入れる
                if (header.DataSize == 0)
                {
                    header.DataMemoryOffset = bssMemoryOffset;
                }

                // Data の終端と BSS の先端との差分を BSS のサイズに足し合わせる
                if (bssMemoryOffset < header.DataMemoryOffset + header.DataSize)
                {
                    throw new ArgumentException(Properties.Resources.Message_InvalidBssSegments);
                }
                header.BssSize += bssMemoryOffset - (header.DataMemoryOffset + header.DataSize);

                // アプリケーションが別途確保するべき BSS セクション用のバッファサイズを NRO のヘッダの BSS サイズとして登録する
                uint reuseSize = AlignUp(header.DataSize, 0x1000) - header.DataSize;
                // DATA セクションの末尾に BSS として利用可能な領域が余っていなければ、BSS の実サイズをそのまま使用する
                if (reuseSize > 0)
                {
                    if (header.BssSize < reuseSize)
                    {
                        header.BssSize = 0;
                    }
                    else
                    {
                        header.BssSize -= reuseSize;
                    }
                }
                header.BssSize = AlignUp(header.BssSize, 0x1000);
            }
            header.DataSize = AlignUp(header.DataSize, 0x1000);
        }

        /// <summary>
        /// NRO ファイルを出力します
        /// </summary>
        /// <param name="fs">出力ファイルストリーム</param>
        public void WriteData(FileStream fs)
        {
            using (MemoryStream ms = new MemoryStream())
            {
                BinaryWriter bw = new BinaryWriter(ms);
                if (header.TextSize > 0)
                {
                    bw.Write(textBinary);
                }

                ms.Seek(header.RoMemoryOffset, SeekOrigin.Begin);
                if (header.RoSize > 0)
                {
                    bw.Write(roBinary);
                }

                ms.Seek(header.DataMemoryOffset, SeekOrigin.Begin);
                if (header.DataSize > 0)
                {
                    bw.Write(dataBinary);
                }

                // NRO ファイルのサイズを 4KB に切り上げる
                if (dataBinary.Length < header.DataSize)
                {
                    ms.Seek(header.DataMemoryOffset + header.DataSize - 1, SeekOrigin.Begin);
                    byte zero = 0;
                    bw.Write(zero);
                }

                header.Size = (uint)ms.Length;
                var buffer = new byte[headerSize];
                var handle = GCHandle.Alloc(buffer, GCHandleType.Pinned);

                try
                {
                    Marshal.StructureToPtr(header, handle.AddrOfPinnedObject(), false);
                }
                finally
                {
                    handle.Free();
                }

                ms.Seek(NroHeaderStartOffset, SeekOrigin.Begin);
                bw.Write(buffer);

                ms.Seek(0, SeekOrigin.Begin);
                ms.CopyTo(fs);
            }
        }

        /// <summary>
        /// NRO ファイルのヘッダ情報を出力します
        /// </summary>
        public void PrintNroHeader()
        {
            Console.Write("signature: ");
            for (int i = 0; i < header.Signature.GetLength(0); i++)
            {
                Console.Write("{0}", (char)header.Signature[i]);
            }
            Console.WriteLine();
            Console.WriteLine("flags: 0x{0:X}", header.Flags);
            Console.WriteLine("text mem offset: 0x{0:X}", header.TextMemoryOffset);
            Console.WriteLine("text file size: 0x{0:X}", header.TextSize);
            Console.WriteLine("ro mem offset: 0x{0:X}", header.RoMemoryOffset);
            Console.WriteLine("ro file size: 0x{0:X}", header.RoSize);
            Console.WriteLine("data mem offset: 0x{0:X}", header.DataMemoryOffset);
            Console.WriteLine("data file size: 0x{0:X}", header.DataSize);
            Console.WriteLine("bss file size: 0x{0:X}", header.BssSize);
            Console.WriteLine("embeded offset: 0x{0:X}", header.EmbededOffset);
            Console.WriteLine("embeded size: 0x{0:X}", header.EmbededSize);
        }

        private uint AlignUp(uint n, uint align)
        {
            return (n + align - 1) & ~(align - 1);
        }
    }
}
