﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.InteropServices;

namespace MakeMeta
{
    [StructLayout(LayoutKind.Sequential)]
    internal struct MetaHeader
    {
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0x4)]
        public byte[] Signature;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0x8)]
        public byte[] Reserved1;
        public byte Flags;
        public byte UnusedData;
        public byte MainThreadPriority;
        public byte MainThreadCoreNumber;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0x4)]
        public byte[] Reserved2;
        public uint SystemResourceSize;
        public uint Version;
        public uint MainThreadStackSize;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0x10)]
        public byte[] Name;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0x10)]
        public byte[] ProductCode;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0x30)]
        public byte[] Reserved3;
        public uint AciOffset;
        public uint AciSize;
        public uint AcidOffset;
        public uint AcidSize;
    }

    internal class Meta
    {
        private MetaHeader Header;

        public const int MaxNameSize = 15;
        public const int MaxProductCodeSize = 15;

        public Meta()
        {
            Header = new MetaHeader();
            Header.Signature = System.Text.Encoding.ASCII.GetBytes("META");
            Header.Reserved1 = new byte[0x8];
            Header.Reserved2 = new byte[0x8];
            Header.Reserved3 = new byte[0x30];
            Header.Name = new byte[0x10];
            Header.ProductCode = new byte[0x10];
        }

        public byte[] ExportBinary()
        {
            uint HeaderSize = (uint)Marshal.SizeOf(typeof(MetaHeader));
            uint alignSize = 0x10;

            Header.AcidOffset = HeaderSize;
            Header.AciOffset = Header.AcidOffset + RoundUp(Header.AcidSize, alignSize);

            uint binarySize = HeaderSize;
            byte[] outputBinary = new byte[binarySize];

            GCHandle gch = GCHandle.Alloc(outputBinary, GCHandleType.Pinned);
            Marshal.StructureToPtr(Header, gch.AddrOfPinnedObject(), false);
            gch.Free();

            return outputBinary;
        }

        public bool Is64BitInstruction
        {
            get
            {
                return (Header.Flags & 1) > 0;
            }
            set
            {
                Header.Flags |= (byte)(value ? 1 : 0);
            }
        }

        public string ProcessAddressSpace
        {
            get
            {
                byte mask = (byte)(((1 << 3) - 1) << 1);
                byte size = (byte)((Header.Flags & mask) >> 1);
                switch (size)
                {
                    case 0:
                        return "AddressSpace32Bit";
                    case 1:
                        return "AddressSpace64BitOld";
                    case 2:
                        return "AddressSpace32BitNoReserved";
                    case 3:
                        return "AddressSpace64Bit";
                    default:
                        throw new ArgumentException(Properties.Resources.Message_InvalidProcessAddressSpaceFlag);
                }
            }
            set
            {
                byte size = 4;
                switch (value)
                {
                    case "AddressSpace32Bit":
                        size = 0;
                        break;
                    case "AddressSpace64BitOld":
                        size = 1;
                        break;
                    case "AddressSpace32BitNoReserved":
                        size = 2;
                        break;
                    case "AddressSpace64Bit":
                        size = 3;
                        break;
                    default:
                        throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidString, "ProcessAddressSpace", value));
                }
                Header.Flags |= (byte)(size << 1);
            }
        }

        public byte MainThreadPriority
        {
            get
            {
                return Header.MainThreadPriority;
            }
            set
            {
                Header.MainThreadPriority = value;
            }
        }

        public byte MainThreadCoreNumber
        {
            get
            {
                return Header.MainThreadCoreNumber;
            }
            set
            {
                Header.MainThreadCoreNumber = value;
            }
        }

        public uint MainThreadStackSize
        {
            get
            {
                return Header.MainThreadStackSize;
            }
            set
            {
                Header.MainThreadStackSize = value;
            }
        }

        public uint SystemResourceSize
        {
            get
            {
                return Header.SystemResourceSize;
            }
            set
            {
                Header.SystemResourceSize = value;
            }
        }

        public uint Version
        {
            get
            {
                return Header.Version;
            }
            set
            {
                Header.Version = value;
            }
        }

        public string Name
        {
            get
            {
                return System.Text.Encoding.UTF8.GetString(Header.Name);
            }
            set
            {
                byte[] name = System.Text.Encoding.UTF8.GetBytes(value);
                if (name.Length > MaxNameSize)
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidStringLength, "Core/Name", MaxNameSize));
                }
                Array.Copy(name, Header.Name, name.Length);
            }
        }

        public string ProductCode
        {
            get
            {
                return System.Text.Encoding.ASCII.GetString(Header.ProductCode);
            }
            set
            {
                if (value.Length > MaxProductCodeSize)
                {
                    throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidStringLength, "Core/ProductCode", MaxProductCodeSize));
                }
                byte[] productCode = System.Text.Encoding.ASCII.GetBytes(value);
                Array.Copy(productCode, Header.ProductCode, productCode.Length);
            }
        }

        public uint AciSize
        {
            get
            {
                return Header.AciSize;
            }
            set
            {
                Header.AciSize = value;
            }
        }

        public uint AcidSize
        {
            get
            {
                return Header.AcidSize;
            }
            set
            {
                Header.AcidSize = value;
            }
        }

        public uint RoundUp(uint target, uint align)
        {
            return ((target + (align - 1)) & ~(align - 1));
        }
    }

    internal class MetaGenerator
    {
        public static Meta Merge(MetaModel.MetaModel meta, DescModel.OutputDescModel desc, uint defaultCoreVersion)
        {
            Meta metaData = new Meta();

            if (meta == null)
            {
                throw new ApplicationException(string.Format(Properties.Resources.Message_NotFoundFile, ".nmeta"));
            }
            else if (desc == null)
            {
                throw new ApplicationException(string.Format(Properties.Resources.Message_NotFoundFile, ".desc"));
            }

            if (desc.Default == null)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_NotFound, ".desc", "Default"));
            }

            desc.Default.CheckReadSuccess();

            metaData.Is64BitInstruction = (meta.Is64BitInstruction != null) ? meta.Is64BitInstructionValue : desc.Default.Is64BitInstructionValue;
            metaData.ProcessAddressSpace = (meta.ProcessAddressSpace != null) ? meta.ProcessAddressSpace : desc.Default.ProcessAddressSpace;
            metaData.MainThreadPriority = (meta.MainThreadPriority != null) ? meta.MainThreadPriorityValue : desc.Default.MainThreadPriorityValue;
            metaData.MainThreadCoreNumber = (meta.MainThreadCoreNumber != null) ? meta.MainThreadCoreNumberValue : desc.Default.MainThreadCoreNumberValue;
            metaData.MainThreadStackSize = (meta.MainThreadStackSize != null) ? meta.MainThreadStackSizeValue : desc.Default.MainThreadStackSizeValue;
            metaData.SystemResourceSize = (meta.SystemResourceSize != null) ? meta.SystemResourceSizeValue : 0;
            metaData.Version = (meta.Version != null) ? meta.VersionValue : defaultCoreVersion;

            byte[] filenameBytes = System.Text.Encoding.UTF8.GetBytes(meta.MetaFileName);
            string filename = System.Text.Encoding.UTF8.GetString(filenameBytes, 0, ((filenameBytes.Length > Meta.MaxNameSize) ? Meta.MaxNameSize : filenameBytes.Length));
            metaData.Name = (meta.Name != null) ? meta.Name : filename;
            metaData.ProductCode = (meta.ProductCode != null) ? meta.ProductCode : string.Empty;

            return metaData;
        }
    }
}
