﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace MakeInitialProgram
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Runtime.Serialization;
    using System.Text;
    using System.Text.RegularExpressions;
    using YamlDotNet.Core;
    using YamlDotNet.Serialization;
    using YamlDotNet.RepresentationModel;

    /// <summary>
    /// パラメータ指定ファイルを表します。
    /// </summary>
    internal sealed class ParameterInfo
    {
        /// <summary>
        /// ParameterInfo クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="fileName">パラメータ指定ファイルのパスです。</param>
        /// <param name="syscallInfo">システムコールの判別に利用するクラスです。</param>
        /// <param name="isTestMode">テスト用のデータを作成するかを示している変数です。</param>
        internal ParameterInfo(string fileName, SystemCallInfo syscallInfo, bool isTestMode)
        {
            var infoFile = ReadYamlFile(fileName, syscallInfo, isTestMode);

            this.FileName = fileName;

            this.Name = GetName(infoFile.Name, InitialProgram.MaxNameLength);

            this.ProgramId = GetProgramId(infoFile.ProgramId);

            this.Version = GetUInt(infoFile.Version, Tag.Version);

            this.StackSize = GetUInt(infoFile.StackSize, Tag.StackSize);

            this.Priority = GetByte(infoFile.Priority, Tag.Priority);

            this.IdealProcessor = GetByte(infoFile.IdealProcessor, Tag.IdealProcessor);

            this.UseSecureMemory = GetBoolean(infoFile.UseSecureMemory, Tag.UseSecureMemory);

            if (infoFile.Capability != null)
            {
                this.Capability = infoFile.Capability.MakeCapability();
            }
            else
            {
                this.Capability = CapabilityParameter.MakeNoneCapability();
            }
        }

        /// <summary>
        /// パラメータ指定ファイルの絶対パスを取得します。
        /// </summary>
        internal string FileName { get; private set; }

        /// <summary>
        /// 初期プログラムの名前を取得します。
        /// </summary>
        internal string Name { get; private set; }

        /// <summary>
        /// 初期プログラムのプログラム ID を取得します。
        /// </summary>
        internal ulong ProgramId { get; private set; }

        /// <summary>
        /// 初期プログラムのバージョンを取得します。
        /// </summary>
        internal uint Version { get; private set; }

        /// <summary>
        /// メインスレッドのスタックサイズを取得します。
        /// </summary>
        internal uint StackSize { get; private set; }

        /// <summary>
        /// メインスレッドの優先度を取得します。
        /// </summary>
        internal byte Priority { get; private set; }

        /// <summary>
        /// メインスレッドの優先プロセッサ番号を取得します。
        /// </summary>
        internal byte IdealProcessor { get; private set; }

        /// <summary>
        /// カーネルケイパビリティを取得します。
        /// </summary>
        internal uint[] Capability { get; private set; }

        /// <summary>
        /// セキュアなメモリ上で実行するかどうかを取得します。
        /// </summary>
        internal bool UseSecureMemory { get; private set; }

        /// <summary>
        /// 現在のオブジェクトを表す文字列を返します。
        /// </summary>
        /// <returns>
        /// 現在のオブジェクトを表す文字列です。
        /// </returns>
        public override string ToString()
        {
            var sb = new StringBuilder();

            sb.AppendLine("Parameter Information:");

            sb.AppendLine(string.Format("  FileName:       {0}", Path.GetFileName(this.FileName)));

            if (this.Name != null)
            {
                sb.AppendLine(string.Format("  Name:           {0}", this.Name));
            }

            sb.AppendLine(string.Format("  ProgramID:      0x{0}", this.ProgramId.ToString("x16")));

            sb.AppendLine(string.Format("  Version:        0x{0}", this.Version.ToString("x8")));

            sb.AppendLine(string.Format("  StackSize:      0x{0}", this.StackSize.ToString("x8")));

            sb.AppendLine(string.Format("  Priority:       {0}", this.Priority));

            sb.Append(string.Format("  IdealProcessor: {0}", this.IdealProcessor));

            sb.Append("\n  Capability:\n");
            foreach (var flag in this.Capability)
            {
                sb.Append(string.Format("    0x{0:X}\n", flag));
            }

            return sb.ToString();
        }

        private static string GetName(string name, int maxLength)
        {
            if (name == null)
            {
                return name;
            }

            if (name.Length > maxLength)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_TooLongString, maxLength, Tag.Name));
            }

            if (Regex.IsMatch(name, @"[^\x20\x21\x23-\x29\x2b-\x2e\d\x3d\x40-\x5b\x5d-\x7b\x7d\x7e]"))
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidString, Tag.Name));
            }

            return name;
        }

        private static ulong GetProgramId(string programId)
        {
            if (programId == null)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_NotDefined, Tag.ProgramId));
            }

            if (!Regex.IsMatch(programId, @"^0x[0-9a-fA-F]{16}$"))
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidValue, Tag.ProgramId));
            }

            return Convert.ToUInt64(programId, 16);
        }

        private static uint GetUInt(string str, string tagName)
        {
            if (str == null)
            {
                throw new ArgumentException(string.Format(Properties.Resources.Message_NotDefined, tagName));
            }

            if ((str.Length <= 10) && Regex.IsMatch(str, @"^(0|[1-9]\d*)$"))
            {
                var value = Convert.ToUInt64(str);

                if (value <= uint.MaxValue)
                {
                    return (uint)value;
                }
            }
            else if ((str.Length <= 10) && Regex.IsMatch(str, @"^0x[0-9a-fA-F]+$"))
            {
                var value = Convert.ToUInt64(str, 16);

                if (value <= uint.MaxValue)
                {
                    return (uint)value;
                }
            }

            throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidUnsignedValue, tagName, 32));
        }

        private static byte GetByte(string str, string tagName)
        {
            if (str == null)
            {
                throw new ArgumentException(string.Format("パラメータ {0} が設定されていません。", tagName));
            }

            if ((str.Length <= 3) && Regex.IsMatch(str, @"^(0|[1-9]\d*)$"))
            {
                var value = Convert.ToUInt16(str);

                if (value <= byte.MaxValue)
                {
                    return (byte)value;
                }
            }

            throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidUnsignedValue, tagName, 8));
        }

        private static bool GetBoolean(string str, string tagName)
        {
            if (str == null)
            {
                throw new ArgumentException(string.Format("パラメータ {0} が設定されていません。", tagName));
            }

            return Convert.ToBoolean(str);
        }


        private static ParameterInfoFile ReadYamlFile(string filePath, SystemCallInfo syscallInfo, bool isTestMode)
        {
            using (var sr = new StreamReader(filePath))
            {
                ParameterInfoFile info = new ParameterInfoFile();
                YamlStream yaml = new YamlStream();
                yaml.Load(sr);

                if (yaml.Documents.Count != 1)
                {
                    throw new ArgumentException("パラメータ指定ファイルのフォーマットが不正です。");
                }

                var rootNode = (YamlMappingNode)yaml.Documents[0].RootNode;
                foreach (var child in rootNode.Children)
                {
                    if (!(child.Key is YamlScalarNode))
                    {
                        throw new ApplicationException(
                            string.Format(Properties.Resources.Message_ParseError, child.ToString()));
                    }

                    string tagName = (child.Key as YamlScalarNode).Value;
                    switch (tagName)
                    {
                        case Tag.Name:
                            info.Name = GetStringFromScalarNode(tagName, child.Value);
                            break;
                        case Tag.ProgramId:
                            info.ProgramId = GetStringFromScalarNode(tagName, child.Value);
                            break;
                        case Tag.Version:
                            info.Version = GetStringFromScalarNode(tagName, child.Value);
                            break;
                        case Tag.StackSize:
                            info.StackSize = GetStringFromScalarNode(tagName, child.Value);
                            break;
                        case Tag.Priority:
                            info.Priority = GetStringFromScalarNode(tagName, child.Value);
                            break;
                        case Tag.IdealProcessor:
                            info.IdealProcessor = GetStringFromScalarNode(tagName, child.Value);
                            break;
                        case Tag.UseSecureMemory:
                            info.UseSecureMemory = GetStringFromScalarNode(tagName, child.Value);
                            break;
                        case Tag.Capability:
                            if (!(child.Value is YamlSequenceNode))
                            {
                                throw new ArgumentException(Properties.Resources.Message_ParseError, "capability");
                            }

                            CapabilityParameter capability = new CapabilityParameter(syscallInfo, isTestMode);
                            capability.ParseCapability(child.Value as YamlSequenceNode);
                            info.Capability = capability;
                            break;
                        default:
                            throw new ArgumentException(string.Format(Properties.Resources.Message_InvalidParameter, tagName));
                    }
                }

                return info;
            }
        }

        private static string GetStringFromScalarNode(string tagName, YamlNode node)
        {
            if (!(node is YamlScalarNode))
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_ParseError, tagName));
            }
            return (node as YamlScalarNode).Value;
        }

        private static class Tag
        {
            internal const string Name = "name";

            internal const string ProgramId = "program_id";

            internal const string Version = "version";

            internal const string StackSize = "stack_size";

            internal const string Priority = "priority";

            internal const string IdealProcessor = "ideal_processor";

            internal const string UseSecureMemory = "use_secure_memory";

            internal const string Capability = "capability";
        }

        private sealed class ParameterInfoFile
        {
            public string Name { get; set; }

            public string ProgramId { get; set; }

            public string Version { get; set; }

            public string StackSize { get; set; }

            public string Priority { get; set; }

            public string IdealProcessor { get; set; }

            public string UseSecureMemory { get; set; }

            public CapabilityParameter Capability { get; set; }
        }
    }
}
