﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using YamlDotNet.Core;
using YamlDotNet.Serialization;
using YamlDotNet.RepresentationModel;
using System.Runtime.CompilerServices;

[assembly: InternalsVisibleTo("MakeInitialProgramTest")]
namespace MakeInitialProgram
{
    internal class CapabilityParameter
    {
        private class FlagParameter
        {
            public uint Value { get; set; }

            public uint NumSignBit { get; set; }

            public uint Flag
            {
                get
                {
                    uint result = 0;
                    uint Sign = (NumSignBit < 32) ? (1U << (int)NumSignBit) - 1 : (uint)0xFFFFFFFF;
                    result = Sign;
                    result |= Value << (int)(NumSignBit + 1);
                    return result;
                }
            }
        }

        private static class CapabilityTag
        {
            internal const string EnableInterrupts = "EnableInterrupts";

            internal const string EnableSystemCalls = "EnableSystemCalls";

            internal const string HandleTableSize = "HandleTableSize";

            internal const string MemoryMap = "MemoryMap";

            // 以下 Test Mode 用
            internal const string Padding = "Padding";

            internal const string Flags = "Flags";
        }

        private static class MemoryMapTag
        {
            internal const string BeginAddress = "BeginAddress";

            internal const string Size = "Size";

            internal const string Permission = "Permission";

            internal const string Type = "Type";
        }

        private static class MemoryMapPermissionTag
        {
            internal const string Ro = "RO";

            internal const string Rw = "RW";
        }

        private static class MemoryMapTypeTag
        {
            internal const string Static = "Static";

            internal const string Io = "IO";
        }

        private enum ParseIntFunc
        {
            ParseIntFunc_10Only,
            ParseIntFunc_16Only,
            ParseIntFunc_Both,
        }

        private const int CapabilityDataSize = 128;
        private const int MaxCapabilityDataNum = CapabilityDataSize / sizeof(uint);

        private const int PageSize = 0x1000;

        private const int PermissionRo = 1;
        private const int PermissionRw = 0;

        private const int TypeStatic = 1;
        private const int TypeIo = 0;

        private List<FlagParameter> Capabilities { get; set; }

        private SystemCallInfo SysCalls { get; set; }

        private bool IsSetEnableInterrupts { get; set; }

        private bool IsSetEnableSystemCalls { get; set; }

        private bool IsSetHandleTableSize { get; set; }

        private bool IsTestMode { get; set; }

        /// <summary>
        /// CapabilityParameter のデフォルトコンストラクタ
        /// パラメータ指定ファイルを使って CapabilityParameter を設定しない場合はこちらを使う
        /// </summary>
        internal CapabilityParameter()
        {
            InitializeParameter();
        }

        /// <summary>
        /// CapabilityParameter のコンストラクタ
        /// </summary>
        /// <param name="syscallInfo">外部ファイルから読み込んだシステムコールの情報</param>
        /// <param name="isTestMode">テストモードかどうか</param>
        internal CapabilityParameter(SystemCallInfo syscallInfo, bool isTestMode)
        {
            InitializeParameter();
            this.SysCalls = syscallInfo;
            this.IsTestMode = isTestMode;
        }

        /// <summary>
        /// 32 ビットの Capaibility の要素データを配列にして返します。
        /// </summary>
        /// <returns>Padding データを含めた Capability のデータ</returns>
        internal uint[] MakeCapability()
        {
            if (this.Capabilities.Count > MaxCapabilityDataNum)
            {
                throw new ArgumentException(Properties.Resources.Message_TooManyCapabilities);
            }

            List<FlagParameter> result = new List<FlagParameter>(this.Capabilities);

            while (result.Count < MaxCapabilityDataNum)
            {
                result.Add(MakePadding());
            }

            return result.Select(flag => flag.Flag).ToArray();
        }

        /// <summary>
        /// Capability を F で埋め尽くしたデータを作ります。
        /// </summary>
        /// <returns>F 埋めされたCapability</returns>
        internal static uint[] MakeNoneCapability()
        {
            CapabilityParameter param = new CapabilityParameter();

            for (int i = 0; i < MaxCapabilityDataNum; i++)
            {
                param.SetPadding();
            }

            return param.MakeCapability();
        }

        /// <summary>
        /// Yamlのノードを解析して、CapabilityParameter のデータを作ります。
        /// </summary>
        /// <param name="node">capability のYamlSequenceNode</param>
        internal void ParseCapability(YamlSequenceNode node)
        {
            foreach (var capability in node.Children)
            {
                if (!(capability is YamlMappingNode))
                {
                    if (IsTestMode)
                    {
                        CheckIsScalarNode(capability);

                        string paddingName = (capability as YamlScalarNode).Value;
                        if (paddingName == CapabilityTag.Padding)
                        {
                            SetPadding();
                            continue;
                        }
                        else
                        {
                            throw new ArgumentException(
                                string.Format(Properties.Resources.Message_InvalidParameter, paddingName));
                        }
                    }
                    else
                    {
                        throw new ArgumentException(
                            string.Format(Properties.Resources.Message_InvalidYmlFormat, capability.Start.Line));
                    }
                }

                var tagNode = (capability as YamlMappingNode);
                if (tagNode.Count() != 1)
                {
                    throw new ArgumentException(
                        string.Format(Properties.Resources.Message_InvalidYmlFormat, tagNode.Start.Line));
                }

                var keys = tagNode.Children.Keys.ToArray();
                CheckIsScalarNode(keys[0]);

                var value = tagNode.Children.Values.ToArray()[0];
                string tagName = (keys[0] as YamlScalarNode).Value;

                switch (tagName)
                {
                    case CapabilityTag.EnableInterrupts:
                        SetEnableInterrupts(value);
                        break;
                    case CapabilityTag.EnableSystemCalls:
                        SetEnableSystemCalls(value);
                        break;
                    case CapabilityTag.HandleTableSize:
                        SetHandleTableSize(value);
                        break;
                    case CapabilityTag.MemoryMap:
                        SetMemoryMap(value);
                        break;
                    case CapabilityTag.Flags:
                        if (!IsTestMode)
                        {
                            throw new ArgumentException(
                                string.Format(Properties.Resources.Message_InvalidParameter, tagName));
                        }

                        SetOtherFlags(value);
                        break;
                    default:
                        throw new ArgumentException(
                            string.Format(Properties.Resources.Message_InvalidParameter, tagName));
                }
            }
        }

        private void InitializeParameter()
        {
            Capabilities = new List<FlagParameter>();
            this.SysCalls = null;
            IsSetEnableInterrupts = false;
            IsSetEnableSystemCalls = false;
            IsSetHandleTableSize = false;
            IsTestMode = false;
        }

        private void SetEnableInterrupts(YamlNode node)
        {
            if (CheckWithTestMode(IsSetEnableInterrupts))
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_AlreadyRegistered, CapabilityTag.EnableInterrupts));
            }

            CheckIsSequenceNode(CapabilityTag.EnableInterrupts, node);

            List<uint> interrupts = new List<uint>();
            foreach (var interruptNode in (node as YamlSequenceNode).Children)
            {
                CheckIsScalarNode(interruptNode);

                var interrupt = (interruptNode as YamlScalarNode).Value;
                var interruptNum = ParseUInt32(CapabilityTag.EnableInterrupts, interrupt, ParseIntFunc.ParseIntFunc_10Only);
                if (interruptNum == 1023)
                {
                    throw new ArgumentException(string.Format(
                        Properties.Resources.Message_PaddingInterruptNumber, CapabilityTag.EnableInterrupts));
                }
                if (interruptNum > 1023)
                {
                    throw new ArgumentException(string.Format(
                        Properties.Resources.Message_InvalidUnsignedValue, CapabilityTag.EnableInterrupts, 10));
                }
                if (CheckWithTestMode(interrupts.Contains(interruptNum)))
                {
                    throw new ArgumentException(string.Format(
                        Properties.Resources.Message_AlreadyRegistered, CapabilityTag.EnableInterrupts));
                }
                interrupts.Add(interruptNum);
            }

            Capabilities.AddRange(MakeEnableInterrupts(interrupts));
            IsSetEnableInterrupts = true;
        }

        private List<FlagParameter> MakeEnableInterrupts(List<uint> interrupts)
        {
            List<FlagParameter> result = new List<FlagParameter>();

            int i = 0;
            if (interrupts.Count % 2 != 0)
            {
                FlagParameter param = new FlagParameter();
                param.Value = interrupts[0] | (1023 << 10);
                param.NumSignBit = 11;
                result.Add(param);
                i++;
            }

            for (; i < interrupts.Count; i += 2)
            {
                FlagParameter param = new FlagParameter();
                param.Value = interrupts[i] | (interrupts[i + 1] << 10);
                param.NumSignBit = 11;
                result.Add(param);
            }

            return result;
        }

        private void SetEnableSystemCalls(YamlNode node)
        {
            if (CheckWithTestMode(IsSetEnableSystemCalls))
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_AlreadyRegistered, CapabilityTag.EnableSystemCalls));
            }

            CheckIsSequenceNode(CapabilityTag.EnableSystemCalls, node);

            List<uint> syscalls = new List<uint>();

            foreach (var syscallNode in (node as YamlSequenceNode))
            {
                CheckIsScalarNode(syscallNode);

                var syscallName = (syscallNode as YamlScalarNode).Value;

                uint syscallNum;

                if (IsTestMode && uint.TryParse(syscallName, out syscallNum))
                {
                    syscalls.Add(syscallNum);
                }
                else
                {
                    if (!SysCalls.HasSystemCall(syscallName))
                    {
                        throw new ArgumentException(
                            string.Format(Properties.Resources.Message_UnknownSystemCall,
                            CapabilityTag.EnableSystemCalls, syscallName));
                    }

                    syscallNum = SysCalls.GetSystemCallNumber(syscallName);
                    if (syscalls.Contains(syscallNum))
                    {
                        throw new ArgumentException(
                            string.Format(Properties.Resources.Message_DuplicatedElement,
                            CapabilityTag.EnableSystemCalls, syscallName));
                    }

                    if (syscallNum >= 128)
                    {
                        throw new ArgumentException(
                            string.Format(Properties.Resources.Message_BiggerVlaue,
                            syscallName, 128));
                    }

                    syscalls.Add(syscallNum);
                }
            }

            Capabilities.AddRange(MakeEnableSystemCalls(syscalls));
            IsSetEnableSystemCalls = true;
        }

        private List<FlagParameter> MakeEnableSystemCalls(List<uint> syscalls)
        {
            const uint MaxSyscall = 24;
            Dictionary<uint, uint> flags = new Dictionary<uint, uint>();

            foreach (var syscallNum in syscalls)
            {
                uint index = syscallNum / MaxSyscall;
                // index は 3bit まで
                if (index >= 8)
                {
                    throw new ArgumentException(
                        string.Format(Properties.Resources.Message_BiggerVlaue, CapabilityTag.EnableSystemCalls, 192));
                }

                var shift = (int)(syscallNum % MaxSyscall);

                if (!flags.ContainsKey(index))
                {
                    flags.Add(index, 0);
                }

                flags[index] |= 1U << shift;
            }

            List<FlagParameter> result = new List<FlagParameter>();

            foreach (var dict in flags)
            {
                FlagParameter flag = new FlagParameter();
                flag.NumSignBit = 4;
                flag.Value = (dict.Key << (int)MaxSyscall) | dict.Value;
                result.Add(flag);
            }

            return result;
        }

        private void SetHandleTableSize(YamlNode node)
        {
            if (CheckWithTestMode(IsSetHandleTableSize))
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_AlreadyRegistered, CapabilityTag.HandleTableSize));
            }

            CheckIsScalarNode(node);

            var tableSize = (node as YamlScalarNode).Value;
            var size = ParseUInt32(CapabilityTag.HandleTableSize, tableSize, ParseIntFunc.ParseIntFunc_10Only);

            Capabilities.Add(MakeHandleTableSize(size));
            IsSetHandleTableSize = true;
        }

        private FlagParameter MakeHandleTableSize(uint tableSize)
        {
            FlagParameter flag = new FlagParameter();

            if (CheckWithTestMode(tableSize >= 1024))
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_BiggerVlaue, CapabilityTag.HandleTableSize, 1024));
            }

            if (tableSize >= (1U << 16))
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_InvalidUnsignedValue, CapabilityTag.HandleTableSize, 16));
            }

            flag.NumSignBit = 15;
            flag.Value = tableSize;

            return flag;
        }

        private void CheckMemoryMapBeginAddress(ulong beginAddr)
        {
            if (CheckWithTestMode(beginAddr >= 1UL << 36))
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_InvalidUnsignedValue, MemoryMapTag.BeginAddress, 36));
            }

            if ((beginAddr & (0xFFF)) > 0)
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_InvalidAlign, MemoryMapTag.BeginAddress, "4KB"));
            }
        }

        private void CheckMemoryMapSize(ulong size)
        {
            if (CheckWithTestMode(size >= 1UL << 32))
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_InvalidUnsignedValue, MemoryMapTag.Size, 32));
            }

            if ((size & (0xFFF)) > 0)
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_InvalidAlign, MemoryMapTag.Size, "4KB"));
            }

            if (CheckWithTestMode(size == 0))
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_Invalid0, MemoryMapTag.Size));
            }
        }

        private void CheckMemoryMapArea(ulong beginAddr, ulong size)
        {
            // マップ領域の終端アドレスが 36 bit 符号なし整数の範囲に収まっているかを調べる
            if (CheckWithTestMode((beginAddr + size - 1) >= 1UL << 36))
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_InvalidMapArea, CapabilityTag.MemoryMap));
            }
        }

        private void SetMemoryMap(YamlNode node)
        {
            CheckIsMappingNode(CapabilityTag.MemoryMap, node);

            // 設定されていない値を見つけるため、ありえない値で初期化する
            const ulong InitAddressValue = 1;
            const ulong InitSizeValue = 1;
            const int InitPermissionValue = -1;
            const int InitTypeValue = -1;

            ulong beginAddr = InitAddressValue;
            ulong size = InitSizeValue;
            int permission = InitPermissionValue;
            int type = InitTypeValue;

            foreach (var mapNode in (node as YamlMappingNode).Children)
            {
                CheckIsScalarNode(mapNode.Key);
                var mapTagName = (mapNode.Key as YamlScalarNode).Value;

                CheckIsScalarNode(mapNode.Value);
                var value = (mapNode.Value as YamlScalarNode).Value;

                switch (mapTagName)
                {
                    case MemoryMapTag.BeginAddress:
                        beginAddr = ParseUInt64(MemoryMapTag.BeginAddress, value, ParseIntFunc.ParseIntFunc_16Only);
                        CheckMemoryMapBeginAddress(beginAddr);
                        break;
                    case MemoryMapTag.Size:
                        size = ParseUInt64(MemoryMapTag.Size, value, ParseIntFunc.ParseIntFunc_16Only);
                        CheckMemoryMapSize(size);
                        break;
                    case MemoryMapTag.Permission:
                        switch (value)
                        {
                            case MemoryMapPermissionTag.Ro:
                                permission = PermissionRo;
                                break;
                            case MemoryMapPermissionTag.Rw:
                                permission = PermissionRw;
                                break;
                            default:
                                throw new ArgumentException(
                                    string.Format(Properties.Resources.Message_InvalidElement,
                                    MemoryMapTag.Permission, value));
                        }
                        break;
                    case MemoryMapTag.Type:
                        switch (value)
                        {
                            case MemoryMapTypeTag.Static:
                                type = TypeStatic;
                                break;
                            case MemoryMapTypeTag.Io:
                                type = TypeIo;
                                break;
                            default:
                                throw new ArgumentException(
                                    string.Format(Properties.Resources.Message_InvalidElement,
                                    MemoryMapTag.Type, value));
                        }
                        break;
                    default:
                        throw new ArgumentException(
                            string.Format(Properties.Resources.Message_InvalidElement,
                            CapabilityTag.MemoryMap, mapTagName));
                }
            }

            if (beginAddr == InitAddressValue)
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_NotDefined,
                    MemoryMapTag.BeginAddress));
            }

            if (size == InitSizeValue)
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_NotDefined,
                    MemoryMapTag.Size));
            }

            if (permission == InitPermissionValue)
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_NotDefined,
                    MemoryMapTag.Permission));
            }

            if (type == InitTypeValue)
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_NotDefined,
                    MemoryMapTag.Type));
            }

            CheckMemoryMapArea(beginAddr, size);

            if (size == PageSize && permission == PermissionRw && type == TypeIo)
            {
                Capabilities.Add(MakeIoMapping(beginAddr));
            }
            else
            {
                Capabilities.AddRange(MakeMapMemory(beginAddr, size, permission, type));
            }
        }

        private List<FlagParameter> MakeMapMemory(ulong beginAddr, ulong size, int permission, int type)
        {
            const int SignBit = 6;
            List<FlagParameter> result = new List<FlagParameter>();
            FlagParameter flag = new FlagParameter();

            CheckMemoryMapBeginAddress(beginAddr);
            CheckMemoryMapSize(size);
            CheckMemoryMapArea(beginAddr, size);

            if (!(permission == PermissionRo || permission == PermissionRw))
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_InvalidValue, MemoryMapTag.Permission));
            }

            if (!(type == TypeStatic || type == TypeIo))
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_InvalidValue, MemoryMapTag.Type));
            }

            flag.NumSignBit = SignBit;
            flag.Value = (uint)(beginAddr >> 12);

            // 24bit以上の値をクリアする
            flag.Value &= (1U << 24) - 1;

            if (permission == PermissionRo)
            {
                flag.Value |= 1U << 24;
            }

            result.Add(flag);

            // For Test Mode
            if (IsTestMode && size == 0)
            {
                return result;
            }

            flag = new FlagParameter();
            flag.NumSignBit = SignBit;

            flag.Value = (uint)(size >> 12);

            // 20bit以上の値をクリアする
            flag.Value &= (1U << 20) - 1;

            if (type == TypeStatic)
            {
                flag.Value |= 1U << 24;
            }

            result.Add(flag);
            return result;
        }

        private FlagParameter MakeIoMapping(ulong beginAddr)
        {
            CheckMemoryMapBeginAddress(beginAddr);
            CheckMemoryMapArea(beginAddr, 0x1000);

            FlagParameter flag = new FlagParameter();
            flag.NumSignBit = 7;
            flag.Value = (uint)(beginAddr >> 12);

            // 24bit以上の値をクリアする
            flag.Value &= (1U << 24) - 1;

            return flag;
        }

        private void SetPadding()
        {
            Capabilities.Add(MakePadding());
        }

        private FlagParameter MakePadding()
        {
            FlagParameter flag = new FlagParameter();
            flag.NumSignBit = 32;
            flag.Value = 0;
            return flag;
        }

        private void SetOtherFlags(YamlNode node)
        {
            CheckIsScalarNode(node);

            var bitValue = (node as YamlScalarNode).Value;
            var flagValue = ParseUInt32(CapabilityTag.Flags, bitValue, ParseIntFunc.ParseIntFunc_16Only);
            Capabilities.Add(MakeOtherFlags(flagValue));
        }

        private FlagParameter MakeOtherFlags(uint flags)
        {
            FlagParameter flag = new FlagParameter();
            flag.NumSignBit = 16;

            if (flags >= 1U << 15)
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_InvalidUnsignedValue, CapabilityTag.Flags, 15));
            }

            flag.Value = flags;

            return flag;
        }

        private bool CheckWithTestMode(bool check)
        {
            return IsTestMode ? false : check;
        }

        private uint ParseUInt32(string tagName, string line, ParseIntFunc func)
        {
            ulong result = ParseUInt64(tagName, line, func);

            if (result > uint.MaxValue)
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_InvalidUnsignedValue, tagName, 32));
            }

            return (uint)result;
        }

        private ulong ParseUInt64(string tagName, string line, ParseIntFunc func)
        {
            ulong result;

            try
            {
                if (line.IndexOf("0x") == 0 && func != ParseIntFunc.ParseIntFunc_10Only)
                {
                    result = ulong.Parse(
                        line.Replace("0x", string.Empty),
                        System.Globalization.NumberStyles.HexNumber,
                        System.Globalization.CultureInfo.InvariantCulture);
                }
                else if (func != ParseIntFunc.ParseIntFunc_16Only)
                {
                    result = ulong.Parse(line);
                }
                else
                {
                    throw new ArgumentException(
                        string.Format(Properties.Resources.Message_InvalidNumberFormat, tagName));
                }
            }
            catch (FormatException)
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_InvalidNumberFormat, tagName));
            }
            catch (OverflowException)
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_InvalidUnsignedValue, tagName, 64));
            }

            return result;
        }

        private void CheckIsMappingNode(string tagName, YamlNode node)
        {
            if (!(node is YamlMappingNode))
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_InvalidMapNode, tagName));
            }
        }

        private void CheckIsSequenceNode(string tagName, YamlNode node)
        {
            if (!(node is YamlSequenceNode))
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_InvalidSequenceNode, tagName));
            }
        }

        private void CheckIsScalarNode(YamlNode node)
        {
            if (!(node is YamlScalarNode))
            {
                throw new ArgumentException(
                    string.Format(Properties.Resources.Message_InvalidScalarNode, node.Start.Line));
            }
        }
    }
}
