﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Numerics;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using Nintendo.Foundation.IO;

namespace MakeAddrRegionHeader
{
    internal static class Logger
    {
        public static void WriteCommandLineError(string text)
        {
            Console.WriteLine("CommandLine: {0}", text);
        }

        public static void WriteError(FilePosition position, MakeAddrRegionHeaderException e)
        {
            Console.WriteLine("{0}: {1}", position, e.Message);
        }
    }

    internal static class PathUtil
    {
        public static string GetRelativePath(string basePath, string relativePath)
        {
            return Path.Combine(Path.GetDirectoryName(basePath), relativePath);
        }
    }

    internal static class ParseUtil
    {
        private static readonly Regex[] NumberRegexes = new Regex[]
        {
            new Regex("^0x[0-9a-fA-F]{8}$", RegexOptions.Compiled),
            new Regex("^0x[0-9a-fA-F]{4}_[0-9a-fA-F]{4}$", RegexOptions.Compiled),
            new Regex("^0x[0-9a-fA-F]{8}_[0-9a-fA-F]{8}$", RegexOptions.Compiled),
            new Regex("^0x[0-9a-fA-F]{4}_[0-9a-fA-F]{4}_[0-9a-fA-F]{4}_[0-9a-fA-F]{4}$", RegexOptions.Compiled),
        };

        public static AddrValue ParseAddrValue(string s, AddrWidth addrWidth)
        {
            if (!IsValidStringAsNumber(s))
            {
                throw new SyntaxErrorException(
                    string.Format(CultureInfo.CurrentCulture, "{0} は、<数値>として無効です。", s));
            }

            ulong value = ParseHexNumber(s);
            if (value > AddrUtil.NumberMax(addrWidth))
            {
                throw new AddrOverflowException(
                    string.Format(CultureInfo.CurrentCulture, "{0} は、<数値>として大きすぎます。", s));
            }
            return new AddrValue(addrWidth, value);
        }

        private static ulong ParseHexNumber(string s)
        {
            return ulong.Parse(s.Substring(2).Replace("_", string.Empty), System.Globalization.NumberStyles.HexNumber, CultureInfo.InvariantCulture);
        }

        private static bool IsValidStringAsNumber(string s)
        {
            return NumberRegexes.Any(r => r.IsMatch(s));
        }
    }

    internal static class ExceptionUtil
    {
        public static void ThrowIfUndefinedEnumArgument(Type enumType, object value, string paramName)
        {
            if (!IsDefinedEnumArgument(enumType, value))
            {
                throw CreateUndefinedEnumArgumentException(enumType, paramName);
            }
        }

        public static bool IsDefinedEnumArgument(Type enumType, object value)
        {
            return Enum.IsDefined(enumType, value);
        }

        public static Exception CreateUndefinedEnumArgumentException(Type enumType, string paramName)
        {
            return new ArgumentOutOfRangeException(
                paramName,
                string.Format(CultureInfo.CurrentCulture, "enum {0} として無効な値です。", enumType.Name));
        }
    }

    internal static class SymbolUtil
    {
        public static string GetRegionValueSymbolName(string regionName, RegionValueType valueType)
        {
            switch (valueType)
            {
                case RegionValueType.Begin:
                    return GetRegionBeginSymbolName(regionName);
                case RegionValueType.End:
                    return GetRegionEndSymbolName(regionName);
                case RegionValueType.Size:
                    return GetRegionSizeSymbolName(regionName);
                default:
                    throw ExceptionUtil.CreateUndefinedEnumArgumentException(typeof(RegionValueType), "valueType");
            }
        }

        public static string GetRegionBeginSymbolName(string regionName)
        {
            return regionName + "_BEGIN";
        }

        public static string GetRegionEndSymbolName(string regionName)
        {
            return regionName + "_END";
        }

        public static string GetRegionSizeSymbolName(string regionName)
        {
            return regionName + "_SIZE";
        }
    }

    internal static class AddrUtil
    {
        public static BigInteger NumberMax(AddrWidth addrWidth)
        {
            switch (addrWidth)
            {
                case AddrWidth.Bit32:
                    return uint.MaxValue;
                case AddrWidth.Bit64:
                    return ulong.MaxValue;
                default:
                    throw new InvalidOperationException();
            }
        }

        public static BigInteger AddrValueEnd(AddrWidth addrWidth)
        {
            return NumberMax(addrWidth) + 1;
        }

        public static bool IsWithinAddrSpace(AddrValue value)
        {
            return 0 <= value.Value && value.Value <= NumberMax(value.AddrWidth);
        }

        public static AddrConstant CreateAddrConstant(string name, AddrValue value)
        {
            return new AddrConstant(name, value);
        }

        public static AddrRegion CreateRegionWithBeginEnd(string name, AddrValue begin, AddrValue end)
        {
            return new AddrRegion(name, begin, end);
        }

        public static AddrRegion CreateRegionWithBeginSize(string name, AddrValue begin, AddrValue size)
        {
            AddrValue end;
            try
            {
                end = begin + size;
            }
            catch (AddrOverflowException)
            {
                throw new AddrOverflowException(
                    string.Format(CultureInfo.CurrentCulture,
                        "領域 {0}: begin + size がオーバーフローしました。({1} + {2})", name, begin, size));
            }

            return new AddrRegion(name, begin, begin + size);
        }

        public static AddrRegion CreateRegionWithShift(string name, AddrRegion baseRegion, Offset offset)
        {
            AddrValue begin;
            try
            {
                begin = baseRegion.Begin + offset;
            }
            catch (AddrOverflowException)
            {
                throw new AddrOverflowException(
                    string.Format(CultureInfo.CurrentCulture,
                        "領域 {0}: begin がオーバーフローしました。 (base region: {1})", name, baseRegion));
            }

            return CreateRegionWithBeginSize(name,
                baseRegion.Begin + offset, baseRegion.Size);
        }

        public static AddrRegion CreateRegionWithInnerOuter(
            string name, AddrRegion baseRegion, InnerOuter innerOuter, Offset offset, AddrValue size)
        {
            AddrValue begin;
            try
            {
                begin = CalcBeginWithInnerOuter(baseRegion, innerOuter, offset, size);
            }
            catch (AddrOverflowException)
            {
                throw new AddrOverflowException(
                    string.Format(CultureInfo.CurrentCulture,
                        "領域 {0}: begin がオーバーフローしました。 (base region: {1})", name, baseRegion));
            }

            return CreateRegionWithBeginSize(name, begin, size);
        }

        private static AddrValue CalcBeginWithInnerOuter(
            AddrRegion baseRegion, InnerOuter innerOuter, Offset offset, AddrValue size)
        {
            if (offset.Sign == Sign.Positive && innerOuter == InnerOuter.Inner)
            {
                return baseRegion.Begin + offset;
            }
            else if (offset.Sign == Sign.Positive && innerOuter == InnerOuter.Outer)
            {
                return baseRegion.End + offset;
            }
            else if (offset.Sign == Sign.Negative && innerOuter == InnerOuter.Inner)
            {
                return baseRegion.End + offset - size;
            }
            else if (offset.Sign == Sign.Negative && innerOuter == InnerOuter.Outer)
            {
                return baseRegion.Begin + offset - size;
            }
            else
            {
                throw new ArgumentException("offset.Sign か innerOuter が不正です。");
            }
        }
    }
}
