﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace HtcDaemon
{
    internal class Message
    {
        // channelId はターゲットが決める。ただし、コントロールチャンネルは0番固定。

        // u16 bodyLength
        // u8  channelId
        // u8  padding
        // u32 ackedSeq // target -> host のみ
        public const int HeaderLength = 8;
        public const int LengthLength = 2;

        public const int LengthPosition = 0;
        public const int ChannelIdPosition = 2;
        public const int PaddingPosition = 3;
        public const int AckedSeqPosition = 4;
        public const int BodyPosition = 8;

        private byte[] data;

        public Message(byte[] data)
        {
            this.data = data;
        }

        public Message(BinaryReader br)
        {
            ushort bodyLength = br.ReadUInt16();
            byte[] data = new byte[HeaderLength + bodyLength];

            int offset = LengthLength;
            while (offset < data.Length)
            {
                int count = br.Read(data, offset, data.Length - offset);
                offset += count;
            }

            this.data = data;
        }

        public byte[] Data { get { return data; } }
        public int Length { get { return data.Length; } }
        public int BodyLength { get { return data.Length - HeaderLength; } }
        public int ChannelId { get { return Data[ChannelIdPosition]; } }
        public uint AckedSeq { get { return BitConverter.ToUInt32(data, AckedSeqPosition); } }

        public BinaryReader MakeBodyReader()
        {
            return new BinaryReader(new MemoryStream(data, HeaderLength, BodyLength, false), Encoding.Unicode);
        }
    }

    internal class MessageWriter
    {
        private byte[] buffer;
        private MemoryStream stream;
        private BinaryWriter writer;

        public MessageWriter(int channelId, int bodyLength)
        {
            Debug.Assert(0 <= bodyLength && bodyLength <= HtcConstants.MessageBodyLengthMax, "MessageWriter に渡されたペイロード長が不正です。 ");

            int messageLength = bodyLength + Message.HeaderLength;
            buffer = new byte[messageLength];
            stream = new MemoryStream(buffer, 0, messageLength);
            writer = new BinaryWriter(stream);

            writer.Write((ushort)bodyLength);
            writer.Write((byte)channelId);
            writer.Write((byte)0);  // padding
            writer.Write((uint)0);  // ackedSeq (target -> host のみ)
        }

        public byte[] GetBuffer()
        {
            return buffer;
        }

        public void Append(byte value)
        {
            writer.Write(value);
        }

        public void Append(short value)
        {
            writer.Write(value);
        }

        public void Append(ushort value)
        {
            writer.Write(value);
        }

        public void Append(int value)
        {
            writer.Write(value);
        }

        public void Append(uint value)
        {
            writer.Write(value);
        }

        public void Append(byte[] value)
        {
            writer.Write(value);
        }

        public void Append(byte[] value, int index, int count)
        {
            writer.Write(value, index, count);
        }

        // 破壊的にメッセージを生成する（message が buffer を所有する）
        public Message MakeMessage()
        {
            return new Message(buffer);
        }
    }
}
