﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;

namespace HtcDaemon.Builtin.HostFs
{
    [Flags]
    internal enum HostFsOpenMode : uint
    {
        OpenMode_Read = (1u << 0),  //!< 読み込みモードで開きます。
        OpenMode_Write = (1u << 1),  //!< 書き込みモードで開きます。
        OpenMode_Create = (1u << 2)
    }

    internal class HostFileSystem
    {
        public const int MaxPathLength = 256;

        public string BasePath { get; private set; }

        public HostFileSystem(string basePath)
        {
            this.BasePath = basePath;
        }

        public HostFile OpenFile(string hostFsPath, HostFsOpenMode openMode)
        {
            return new HostFile(GetRealPath(hostFsPath), openMode);
        }

        public HostDirectory OpenDirectory(string hostFsPath, HostFsOpenMode openMode)
        {
            return new HostDirectory(GetRealPath(hostFsPath));
        }

        public bool FileExists(string hostFsPath)
        {
            return File.Exists(GetRealPath(hostFsPath));
        }

        public bool DirectoryExists(string hostFsPath)
        {
            return Directory.Exists(GetRealPath(hostFsPath));
        }

        public void CreateDirectory(string hostFsPath)
        {
            Directory.CreateDirectory(GetRealPath(hostFsPath));
        }

        public void DeleteFile(string hostFsPath)
        {
            File.Delete(GetRealPath(hostFsPath));
        }

        public void DeleteDirectory(string hostFsPath)
        {
            string path = GetRealPath(hostFsPath);

            try
            {
                Directory.Delete(path);
            }
            catch
            {
                // TODO: 競合条件を修正する
                if (Directory.EnumerateFileSystemEntries(path).Any())
                {
                    throw new DirectoryNotEmptyException();
                }
                else
                {
                    throw;
                }
            }
        }

        public void MoveFile(string currentHostFsPath, string newHostFsPath)
        {
            string currentPath = GetRealPath(currentHostFsPath);
            string newPath = GetRealPath(newHostFsPath);

            // TODO: 競合条件を修正する
            if (!File.Exists(currentPath))
            {
                throw new FileNotFoundException();
            }

            MoveEntry(currentPath, newPath);
        }

        public void MoveDirectory(string currentHostFsPath, string newHostFsPath)
        {
            string currentPath = GetRealPath(currentHostFsPath);
            string newPath = GetRealPath(newHostFsPath);

            // TODO: 競合条件を修正する
            if (!Directory.Exists(currentPath))
            {
                throw new FileNotFoundException();
            }

            MoveEntry(currentPath, newPath);
        }

        private static void MoveEntry(string currentPath, string newPath)
        {
            try
            {
                // Directory.Move はファイルも移動することができます
                Directory.Move(currentPath, newPath);
            }
            catch (IOException)
            {
                // TODO: 競合条件を修正する
                if (File.Exists(newPath) || Directory.Exists(newPath))
                {
                    throw new PathAlreadyExistsException();
                }
                else
                {
                    throw;
                }
            }
        }

        private string GetRealPath(string hostFsPath)
        {
            return Path.Combine(BasePath, hostFsPath);
        }
    }

    internal class HostFile : IDisposable
    {
        private FileStream stream;
        private bool disposed;

        public HostFile(string path, HostFsOpenMode openMode)
        {
            FileAccess access;
            FileShare share;
            switch (openMode & (HostFsOpenMode.OpenMode_Read | HostFsOpenMode.OpenMode_Write))
            {
                case HostFsOpenMode.OpenMode_Read:
                    access = FileAccess.Read;
                    share = FileShare.Read;
                    break;
                case HostFsOpenMode.OpenMode_Write:
                    access = FileAccess.Write;
                    share = FileShare.None;
                    break;
                case HostFsOpenMode.OpenMode_Read | HostFsOpenMode.OpenMode_Write:
                    access = FileAccess.ReadWrite;
                    share = FileShare.None;
                    break;
                default:
                    access = FileAccess.Read;
                    share = FileShare.Read;
                    break;
            }

            FileMode mode;
            if ((openMode & (HostFsOpenMode.OpenMode_Write | HostFsOpenMode.OpenMode_Create)) == (HostFsOpenMode.OpenMode_Write | HostFsOpenMode.OpenMode_Create))
            {
                mode = FileMode.OpenOrCreate;
            }
            else
            {
                mode = FileMode.Open;
            }

            this.stream = File.Open(path, mode, access, share);
        }

        ~HostFile()
        {
            Dispose(false);
        }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        public void Close()
        {
            this.stream.Close();
        }

        public int Read(long position, byte[] buffer, int offset, int size)
        {
            if (this.stream.Position != position)
            {
                this.stream.Seek(position, SeekOrigin.Begin);
            }
            return this.stream.Read(buffer, offset, size);
        }

        public void Write(long position, byte[] buffer, int offset, int size, bool flush)
        {
            if (this.stream.Position != position)
            {
                this.stream.Seek(position, SeekOrigin.Begin);
            }

            this.stream.Write(buffer, offset, size);

            if (flush)
            {
                this.stream.Flush();
            }
        }

        public long GetSize()
        {
            return this.stream.Length;
        }

        public void SetSize(long size)
        {
            this.stream.SetLength(size);
        }

        public void Flush()
        {
            this.stream.Flush();
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    this.stream.Dispose();
                }
                // ここでアンマネージドリソースを解放する
                disposed = true;
            }
        }
    }

    internal class HostDirectory : IDisposable
    {
        private DirectoryInfo directoryInfo;
        private IEnumerator<FileSystemInfo> emumerator;

        private bool disposed;

        public HostDirectory(string path)
        {
            directoryInfo = new DirectoryInfo(path);
            emumerator = directoryInfo.EnumerateFileSystemInfos().GetEnumerator();
        }

        ~HostDirectory()
        {
            Dispose(false);
        }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        public void Close()
        {
            Dispose(false);
        }

        public HostFileSystemEntry[] Read(int count)
        {
            var result = new List<HostFileSystemEntry>();

            while (result.Count < count && this.emumerator.MoveNext())
            {
                try
                {
                    var entry = new HostFileSystemEntry(this.emumerator.Current);
                    result.Add(entry);
                }
                // HostFsEntry に変換できなかったのでスキップ
                catch (ArgumentException)
                {
                }
                catch (IOException)
                {
                }
            }

            return result.ToArray();
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    emumerator.Dispose();
                }
                // ここでアンマネージドリソースを解放する
                disposed = true;
            }
        }
    }

    internal class HostFileSystemEntry
    {
        public const int TransferSize = 8 + 1 * 4 + 2 + HostFileSystem.MaxPathLength;
        public string Name { get; private set; }
        public bool IsDirectory { get; private set; }
        public bool IsHidden { get; private set; }
        public bool IsArchive { get; private set; }
        public bool IsReadOnly { get; private set; }
        public long Size { get; private set; }

        private byte[] multiByteNameBuffer = new byte[HostFileSystem.MaxPathLength];
        private int multiByteNameLength;

        public HostFileSystemEntry(FileSystemInfo fileSystemInfo)
        {
            this.Name = fileSystemInfo.Name;
            multiByteNameLength = Encoding.Default.GetBytes(Name, 0, Name.Length, multiByteNameBuffer, 0);

            this.IsDirectory = (fileSystemInfo.Attributes & FileAttributes.Directory) != 0;
            this.IsHidden = (fileSystemInfo.Attributes & FileAttributes.Hidden) != 0;
            this.IsArchive = (fileSystemInfo.Attributes & FileAttributes.Archive) != 0;
            this.IsReadOnly = (fileSystemInfo.Attributes & FileAttributes.ReadOnly) != 0;

            if (this.IsDirectory)
            {
                this.Size = 0;
            }
            else
            {
                this.Size = ((FileInfo)fileSystemInfo).Length;
            }
        }

        // Entry:
        //   int64_t    size;
        //   bool       isDirectory;
        //   bool       isHidden;
        //   bool       isArchive;
        //   bool       isReadOnly;
        //   int16_t    nameLength;
        //   char       name[HostFileSystem.MaxPathLength];
        public void Write(BinaryWriter writer)
        {
            writer.Write(Size);
            writer.Write(IsDirectory);
            writer.Write(IsHidden);
            writer.Write(IsArchive);
            writer.Write(IsReadOnly);
            writer.Write((short)multiByteNameLength);
            writer.Write(multiByteNameBuffer);
        }
    }
}
