﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.FsFileCacheSimulator.FileSystem
{
    internal class AccessLog
    {
        private IEnumerable<Entry> m_AccessLogEntries;

        public AccessLog(StreamReader reader)
        {
            m_AccessLogEntries = ParseLogs(AccessLogSerializer.Deserialize(reader)).ToArray();
        }
        public AccessLog(IEnumerable<IReadOnlyDictionary<object, object>> serializedAccessLogs)
        {
            m_AccessLogEntries = ParseLogs(serializedAccessLogs).ToArray();
        }
        public AccessLog(IEnumerable<Entry> entries)
        {
            m_AccessLogEntries = entries;
        }

        public IEnumerable<Entry> Entries => m_AccessLogEntries;

        public IEnumerable<IReadOnlyDictionary<object, object>> ToSerializableLogs()
        {
            yield return new Dictionary<object, object>() { { "sdk_version", "0.0.0" }, { "spec", "NX" } };

            foreach (var entry in m_AccessLogEntries)
            {
                yield return entry.ToSerializableLog();
            }
        }

        public ulong StartTimeMillisecond => m_AccessLogEntries.FirstOrDefault()?.Start ?? 0;
        public ulong EndTimeMillisecond => m_AccessLogEntries.LastOrDefault()?.End ?? 0;

        private IEnumerable<Entry> ParseLogs(IEnumerable<IReadOnlyDictionary<object, object>> serializedAccessLogs)
        {
            foreach (var serializedAccessLog in serializedAccessLogs)
            {
                if (serializedAccessLog.ContainsKey("function"))
                {
                    switch (serializedAccessLog["function"] as string)
                    {
                        case "MountRom":
                            yield return new MountRomEntry(serializedAccessLog);
                            break;
                        case "MountSaveData":
                            yield return new MountSaveDataEntry(serializedAccessLog);
                            break;
                        case "MountCacheStorage":
                            yield return new MountCacheStorageEntry(serializedAccessLog);
                            break;
                        case "MountTemporaryStorage":
                            yield return new MountTemporaryStorageEntry(serializedAccessLog);
                            break;
                        case "Unmount":
                            yield return new UnmountEntry(serializedAccessLog);
                            break;
                        case "OpenFile":
                            yield return new OpenFileEntry(serializedAccessLog);
                            break;
                        case "CloseFile":
                            yield return new CloseFileEntry(serializedAccessLog);
                            break;
                        case "ReadFile":
                            yield return new ReadFileEntry(serializedAccessLog);
                            break;
                        case "WriteFile":
                            yield return new WriteFileEntry(serializedAccessLog);
                            break;
                        default:
                            yield return new DontCareEntry(serializedAccessLog);
                            break;
                    }
                }
            }
        }

        public abstract class Entry
        {
            public Entry(IReadOnlyDictionary<object, object> serializedLog)
            {
                Start = ulong.Parse((string)serializedLog["start"]);
                End = ulong.Parse((string)serializedLog["end"]);
                Result = uint.Parse(((string)serializedLog["result"]).Substring(2), NumberStyles.AllowHexSpecifier);
                Handle = ulong.Parse(((string)serializedLog["handle"]).Substring(2), NumberStyles.AllowHexSpecifier);
            }
            public Entry(ulong start, ulong end, uint result, ulong handle)
            {
                Start = start;
                End = end;
                Result = result;
                Handle = handle;
            }

            public ulong Start { get; }
            public ulong End { get; }
            public uint Result { get; }
            public ulong Handle { get; }

            public bool IsResultSuccess => Result == 0;

            public IReadOnlyDictionary<object, object> ToSerializableLog()
            {
                var ret = new Dictionary<object, object>();
                ret.Add("start", Start);
                ret.Add("end", End);
                ret.Add("result", $"0x{Result:x8}");
                ret.Add("handle", $"0x{Handle:x16}");
                AddEntrySpecificValues(ret);
                return ret;
            }

            protected abstract void AddEntrySpecificValues(IDictionary<object, object> dictionary);
        }

        public class MountRomEntry : Entry
        {
            public MountRomEntry(IReadOnlyDictionary<object, object> serializedLog)
                : base(serializedLog)
            {
                Name = (string)serializedLog["name"];
            }
            public MountRomEntry(ulong start, ulong end, uint result, ulong handle, string name)
                : base(start, end, result, handle)
            {
                Name = name;
            }

            public string Name { get; }

            protected override void AddEntrySpecificValues(IDictionary<object, object> dictionary)
            {
                dictionary.Add("function", "MountRom");
                dictionary.Add("name", Name);
            }
        }

        public class MountSaveDataEntry : Entry
        {
            public MountSaveDataEntry(IReadOnlyDictionary<object, object> serializedLog)
                : base(serializedLog)
            {
                Name = (string)serializedLog["name"];
                UserId = (string)serializedLog["userid"];
            }
            public MountSaveDataEntry(ulong start, ulong end, uint result, ulong handle, string name, string userId)
                : base(start, end, result, handle)
            {
                Name = name;
                UserId = userId;
            }

            public string Name { get; }
            public string UserId { get; }

            protected override void AddEntrySpecificValues(IDictionary<object, object> dictionary)
            {
                dictionary.Add("function", "MountSaveData");
                dictionary.Add("name", Name);
                dictionary.Add("userid", UserId);
            }
        }

        public class MountCacheStorageEntry : Entry
        {
            public MountCacheStorageEntry(IReadOnlyDictionary<object, object> serializedLog)
                : base(serializedLog)
            {
                Name = (string)serializedLog["name"];
            }
            public MountCacheStorageEntry(ulong start, ulong end, uint result, ulong handle, string name)
                : base(start, end, result, handle)
            {
                Name = name;
            }

            public string Name { get; }

            protected override void AddEntrySpecificValues(IDictionary<object, object> dictionary)
            {
                dictionary.Add("function", "MountCacheStorage");
                dictionary.Add("name", Name);
            }
        }

        public class MountTemporaryStorageEntry : Entry
        {
            public MountTemporaryStorageEntry(IReadOnlyDictionary<object, object> serializedLog)
                : base(serializedLog)
            {
                Name = (string)serializedLog["name"];
            }
            public MountTemporaryStorageEntry(ulong start, ulong end, uint result, ulong handle, string name)
                : base(start, end, result, handle)
            {
                Name = name;
            }

            public string Name { get; }

            protected override void AddEntrySpecificValues(IDictionary<object, object> dictionary)
            {
                dictionary.Add("function", "MountTemporaryStorage");
                dictionary.Add("name", Name);
            }
        }

        public class UnmountEntry : Entry
        {
            public UnmountEntry(IReadOnlyDictionary<object, object> serializedLog)
                : base(serializedLog)
            {
                Name = (string)serializedLog["name"];
            }
            public UnmountEntry(ulong start, ulong end, uint result, ulong handle, string name)
                : base(start, end, result, handle)
            {
                Name = name;
            }

            public string Name { get; }

            protected override void AddEntrySpecificValues(IDictionary<object, object> dictionary)
            {
                dictionary.Add("function", "Unmount");
                dictionary.Add("name", Name);
            }
        }

        public class OpenFileEntry : Entry
        {
            public OpenFileEntry(IReadOnlyDictionary<object, object> serializedLog)
                : base(serializedLog)
            {
                Path = (string)serializedLog["path"];
            }
            public OpenFileEntry(ulong start, ulong end, uint result, ulong handle, string path)
                : base(start, end, result, handle)
            {
                Path = path;
            }

            public string Path { get; }

            protected override void AddEntrySpecificValues(IDictionary<object, object> dictionary)
            {
                dictionary.Add("function", "OpenFile");
                dictionary.Add("path", Path);
            }
        }

        public class CloseFileEntry : Entry
        {
            public CloseFileEntry(IReadOnlyDictionary<object, object> serializedLog)
                : base(serializedLog)
            {
            }
            public CloseFileEntry(ulong start, ulong end, uint result, ulong handle)
                : base(start, end, result, handle)
            {
            }

            protected override void AddEntrySpecificValues(IDictionary<object, object> dictionary)
            {
                dictionary.Add("function", "CloseFile");
            }
        }

        public class ReadFileEntry : Entry
        {
            public ReadFileEntry(IReadOnlyDictionary<object, object> serializedLog)
                : base(serializedLog)
            {
                Offset = ulong.Parse((string)serializedLog["offset"]);
                Size = ulong.Parse((string)serializedLog["size"]);
            }
            public ReadFileEntry(ulong start, ulong end, uint result, ulong handle, ulong offset, ulong size)
                : base(start, end, result, handle)
            {
                Offset = offset;
                Size = size;
            }

            public ulong Offset { get; }
            public ulong Size { get; }

            protected override void AddEntrySpecificValues(IDictionary<object, object> dictionary)
            {
                dictionary.Add("function", "ReadFile");
                dictionary.Add("offset", Offset);
                dictionary.Add("size", Size);
            }
        }

        public class WriteFileEntry : Entry
        {
            public WriteFileEntry(IReadOnlyDictionary<object, object> serializedLog)
                : base(serializedLog)
            {
                Offset = ulong.Parse((string)serializedLog["offset"]);
                Size = ulong.Parse((string)serializedLog["size"]);
            }
            public WriteFileEntry(ulong start, ulong end, uint result, ulong handle, ulong offset, ulong size)
                : base(start, end, result, handle)
            {
                Offset = offset;
                Size = size;
            }

            public ulong Offset { get; }
            public ulong Size { get; }

            protected override void AddEntrySpecificValues(IDictionary<object, object> dictionary)
            {
                dictionary.Add("function", "WriteFile");
                dictionary.Add("offset", Offset);
                dictionary.Add("size", Size);
            }
        }

        public class DontCareEntry : Entry
        {
            private IReadOnlyDictionary<object, object> m_SerializedLog;

            public DontCareEntry(IReadOnlyDictionary<object, object> serializedLog)
                : base(serializedLog)
            {
                m_SerializedLog = serializedLog;
            }

            protected override void AddEntrySpecificValues(IDictionary<object, object> dictionary)
            {
                foreach (var kv in m_SerializedLog)
                {
                    if (!dictionary.ContainsKey(kv.Key))
                    {
                        dictionary.Add(kv.Key, kv.Value);
                    }
                }
            }
        }
    }
}
