﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;

namespace DeviceTreeDataCollector
{
    public class DeviceTreeProperty
    {
        public enum PropertyType
        {
            AddressRangeList,
            IntegerList,
            StringList,
            Phandle,
        }

        public static PropertyType GetPropertyType(string typeName)
        {
            switch (typeName)
            {
                case "AddressRange":
                case "AddressRangeList":
                    return PropertyType.AddressRangeList;
                case "Integer":
                case "IntegerList":
                    return PropertyType.IntegerList;
                case "String":
                case "StringList":
                    return PropertyType.StringList;
                case "Phandle":
                    return PropertyType.Phandle;
                default:
                    throw new ArgumentException($"Invalid property type {typeName} was specified.");
            }
        }

        public readonly string Name;
        public readonly PropertyType Type;

        private List<object> Values;

        public DeviceTreeProperty(string name, PropertyType type)
        {
            Name = name;
            Type = type;
            Values = new List<object>();
        }

        public void Add<T>(T value)
        {
            CheckType<T>();
            Values.Add(value);
        }

        public IEnumerable<T> GetValues<T>()
        {
            CheckType<T>();
            List<T> actualValues = Values.ConvertAll(x => (T)x);
            return (IEnumerable<T>)actualValues;
        }

        private static Type GetValueType(PropertyType type)
        {
            switch (type)
            {
                case PropertyType.AddressRangeList:
                    return typeof(AddressRange);
                case PropertyType.IntegerList:
                    return typeof(int);
                case PropertyType.StringList:
                    return typeof(string);
                case PropertyType.Phandle:
                    return typeof(Phandle);
                default:
                    throw new InvalidOperationException();
            }
        }

        private void CheckType<T>()
        {
            if (typeof(T) != GetValueType(Type))
            {
                throw new InvalidOperationException($"Value type did not match: {typeof(T)} != {GetValueType(Type)}");
            }
        }
    }

    public class AddressRange : IComparable, IEquatable<AddressRange>
    {
        public readonly ulong BeginAddress;
        public readonly ulong Size;

        public AddressRange(ulong beginAddress, ulong size)
        {
            BeginAddress = beginAddress;
            Size = size;
        }

        public bool Equals(AddressRange other)
        {
            if (ReferenceEquals(other, null)) { return false; }
            if (ReferenceEquals(this, other)) { return true; }
            return BeginAddress.Equals(other.BeginAddress) && Size.Equals(other.Size);
        }

        public override int GetHashCode()
        {
            return BeginAddress.GetHashCode() ^ Size.GetHashCode();
        }

        // IComparable<T>.CompareTo と比較演算子オーバーロード (Sort のため)
        public int CompareTo(object obj)
        {
            if (obj == null) { throw new ArgumentNullException("obj was null"); }
            AddressRange other = obj as AddressRange;
            if (other == null) { throw new ArgumentException("obj was not AddressRange"); }

            // BeginAddress で昇順
            if (this.BeginAddress < other.BeginAddress)
            {
                return -1;
            }
            else if (this.BeginAddress > other.BeginAddress)
            {
                return 1;
            }

            // BeginAddress が同じ場合 Size で昇順
            if (this.Size < other.Size)
            {
                return -1;
            }
            else if (this.Size > other.Size)
            {
                return 1;
            }

            // 一致
            return 0;
        }
        public static bool operator <(AddressRange x, AddressRange y)
        {
            if (x == null) { throw new ArgumentNullException("x was null"); }
            return x.CompareTo(y) < 0;
        }
        public static bool operator <=(AddressRange x, AddressRange y)
        {
            if (x == null) { throw new ArgumentNullException("x was null"); }
            return x.CompareTo(y) <= 0;
        }
        public static bool operator >(AddressRange x, AddressRange y)
        {
            return y < x;
        }
        public static bool operator >=(AddressRange x, AddressRange y)
        {
            return y <= x;
        }

        /// <summary>
        /// 包含の判定
        /// </summary>
        /// <param name="other"></param>
        /// <returns>
        /// other を包含または一致するとき true
        /// </returns>
        public bool Includes(AddressRange other)
        {
            if (other == null) { throw new ArgumentNullException("other was null"); }

            return ((this.BeginAddress <= other.BeginAddress) &&
                (this.BeginAddress + this.Size >= other.BeginAddress + other.Size)) ? true : false;
        }

        public override string ToString()
        {
            return string.Format("0x{0:X16}:0x{1:X16}", BeginAddress, Size);
        }
    }

    public class Phandle
    {
        private readonly uint PhandleValue;
        public Phandle(uint value)
        {
            PhandleValue = value;
        }
        public static explicit operator uint(Phandle val)
        {
            return val.PhandleValue;
        }
    }
}
