﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Json;
using System.IO;

namespace MakeDataDocument
{
    [DataContract]
    public class DataSeries
    {
        [DataMember(Name = "key", Order = 0)]
        public string Key { get; set; }

        [DataMember(Name = "values", Order = 1)]
        public List<Dictionary<string, object>> Values { get; set; }
    }

    public class DataDocumentGenerator
    {
        public DataDocumentGenerator(DataDocumentDefinition definition)
        {
            m_Definition = definition;
        }

        public List<DataSeries> GenerateVisualizableData(IEnumerable<Dictionary<string, object>> datalets)
        {
            var ret = new Dictionary<string, List<Dictionary<string, object>>>();

            foreach (var datalet in datalets)
            {
                if (m_Definition.CheckAcceptability(datalet))
                {
                    var key = datalet[m_Definition.DataSampleDefinition.Key];

                    if (!ret.ContainsKey(key.ToString()))
                    {
                        ret[key.ToString()] = new List<Dictionary<string, object>>();
                    }

                    ret[key.ToString()].Add(new Dictionary<string, object> {
                        {"x", datalet[m_Definition.DataSampleDefinition.X] },
                        {"y", datalet[m_Definition.DataSampleDefinition.Y] }
                    });
                }
            }

            return (from series in ret
                    select new DataSeries
                    {
                        Key = series.Key,
                        Values = series.Value
                    }).ToList();
        }

        public string GenerateVisualizableDataAsJsonText()
        {
            var settings = new DataContractJsonSerializerSettings();
            settings.UseSimpleDictionaryFormat = true;

            var serializer = new DataContractJsonSerializer(typeof(List<DataSeries>), settings);
            var stream = new MemoryStream();
            serializer.WriteObject(stream, GenerateVisualizableData(DataletUtility.EnumerateDataletsFromFiles(DataletUtility.EnumerateDataFiles(m_Definition.SourceDirectory))));

            return Encoding.UTF8.GetString(stream.ToArray());
        }

        public void GenerateVisualizableDataAsJsonFile()
        {
            var outputPath = GetJsonOutputPath();

            Directory.CreateDirectory(Path.GetDirectoryName(outputPath));

            using (var writer = new StreamWriter(outputPath))
            {
                writer.Write(GenerateVisualizableDataAsJsonText());
            }
        }

        private string GetJsonOutputPath()
        {
            return Path.Combine(m_Definition.OutputDirectory, m_Definition.Name + ".json");
        }

        private DataDocumentDefinition m_Definition;
    }
}
