﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using Nintendo.Foundation.IO;
using ContentsUploader.Assistants;

namespace ContentsUploader.Commands
{
    using static Constants;
    using static Models.Pms;

    public class RegisterTitleCommand : PricingCommand
    {
        //! nsp 指定
        [CommandLineOption('s', "input", DefaultValue = "", Description = "The file/derctory path of the nsp that you want to register-title.")]
        public override string NspPathOption { get; set; }

        //! 単体 ID 指定
        [CommandLineOption("application-id", DefaultValue = "", Description = "input application id")]
        public override string ApplicationIdOption { get; set; }

        [CommandLineOption("update-notice-flag-disabled", DefaultValue = false, Description = "Update notice flag be disabled")]
        public bool IsUpdateNoticeFlagDisabled { get; set; }

        [CommandLineOption("force-update-flag-disabled", DefaultValue = false, Description = "Force update flag be disabled")]
        public bool IsForceUpdateFlagDisabled { get; set; }

        // 廃止されたオプション
        [CommandLineOption("sort-name", DefaultValue = "", IsHidden = true, Description = "item sort name")]
        public string SortNameOption { get; set; }

        // 更新通知の可否判定
        private bool IsUpdateNotificationEnabled
        {
            get
            {
                return !IsUpdateNoticeFlagDisabled;
            }
        }

        // 強制更新の可否判定
        private bool IsForcedUpdateEnabled
        {
            get
            {
                return !IsForceUpdateFlagDisabled;
            }
        }

        public override void Run()
        {
            Run("Register Title", Mode.IsTokenRequired);
        }

        protected override bool ValidateOptions()
        {
            var name = Naming.CreateTitle(InitialCodeOption, FormalNameOption);
            var valid = name.IsValid;
            valid &= Setting.Current.SetupNaming(name);
            valid &= Setting.Current.SetupTitleLanguages(LanguageOption);
            valid &= Setting.Current.SetupTitleCountries(TargetCountryOption);
            valid &= SetupSaleOperations();
            return valid;
        }

        //! 実行処理 ->
        protected override bool RunByDirectory()
        {
            var path = string.Empty;
            if (!ToolUtility.ConvertToAbsoluteDirectoryPath(out path, NspPathOption))
            {
                Log.WriteLine($"Error: Directory not found. Please check the path \"{path}\".");
                return false;
            }

            // Rom 登録
            var shop = new ShopHelper(Setting.Current);
            var result = new RegisterRomResult();
            if (!shop.RegisterRomByDirectory(out result, path, IsUpdateNotificationEnabled, IsForcedUpdateEnabled))
            {
                return false;
            }

            // Patch のみの場合、以降の Title API は使わない
            if (result.IsPatchOnly())
            {
                return true;
            }

            // Aoc 登録
            var aocNsUids = new NsUidMap();
            if (result.Aocs.Count > 0)
            {
                if (!RegisterAocByContentMetaIdList(out aocNsUids, result.ApplicationId, result.Aocs))
                {
                    return false;
                }
                if (!RegisterPriceByNsUid("aoc", aocNsUids))
                {
                    return false;
                }
            }

            // Title 登録
            var nsUid = string.Empty;
            if (!RegisterTitleByContentMeta(out nsUid, result.ApplicationId, aocNsUids, result.IsAppContained()))
            {
                return false;
            }
            if (!RegisterPriceByNsUid("title", nsUid, true, result.IsAppContained()))
            {
                return false;
            }
            WriteNsUidList(nsUid, result.ApplicationId, aocNsUids);
            return true;
        }

        protected override bool RunByFile()
        {
            var path = string.Empty;
            if (!ToolUtility.ConvertToAbsoluteNspFilePath(out path, NspPathOption))
            {
                Log.WriteLine($"Error: Nsp file not found. Please check the path \"{path}\".");
                return false;
            }

            // Rom 登録
            var shop = new ShopHelper(Setting.Current);
            var result = new RegisterRomResult();
            if (!shop.RegisterRomByFile(out result, path, IsUpdateNotificationEnabled, IsForcedUpdateEnabled))
            {
                return false;
            }

            // Patch のみの場合、以降の Title API は使わない
            if (result.IsPatchOnly())
            {
                return true;
            }

            // Aoc 登録
            var aocNsUids = new NsUidMap();
            if (result.Aocs.Count > 0)
            {
                if (!RegisterAocByContentMetaIdList(out aocNsUids, result.ApplicationId, result.Aocs))
                {
                    return false;
                }
                if (!RegisterPriceByNsUid("aoc", aocNsUids))
                {
                    return false;
                }
            }

            // Title 登録
            var nsUid = string.Empty;
            if (!RegisterTitleByContentMeta(out nsUid, result.ApplicationId, aocNsUids, result.IsAppContained()))
            {
                return false;
            }
            if (!RegisterPriceByNsUid("title", nsUid, true, result.IsAppContained()))
            {
                return false;
            }
            WriteNsUidList(nsUid, result.ApplicationId, aocNsUids);
            return true;
        }

        protected override bool RunByApplicationId(Id64 applicationId)
        {
            var shop = new ShopHelper(Setting.Current);
            if (!shop.RegisterRomByContentMeta(applicationId, ContentMetaType.Application, IsUpdateNotificationEnabled, IsForcedUpdateEnabled))
            {
                return false;
            }

            var nsUid = string.Empty;
            var aocNsUids = new NsUidMap();
            if (!RegisterTitleByContentMeta(out nsUid, applicationId, aocNsUids, true))
            {
                return false;
            }
            if (!RegisterPriceByNsUid("title", nsUid, true, true))
            {
                return false;
            }
            WriteNsUidList(nsUid, applicationId, aocNsUids);
            return true;
        }
        //! <-

        // Aoc 登録処理 ->
        private bool RegisterAocByContentMetaIdList(out NsUidMap aocNsUids, Id64 applicationId, List<Id64> contentMetaIds)
        {
            aocNsUids = new NsUidMap();
            foreach (var contentMetaId in contentMetaIds)
            {
                var nsUid = string.Empty;
                if (!RegisterAocByContentMetaId(out nsUid, applicationId, contentMetaId))
                {
                    return false;
                }
                aocNsUids.Add(contentMetaId, nsUid);
            }
            return true;
        }

        private bool RegisterAocByContentMetaId(out string nsUid, Id64 applicationId, Id64 contentMetaId)
        {
            nsUid = string.Empty;

            // Aoc の nsUid を特定
            var shop = new ShopHelper(Setting.Current);
            if (!shop.ConvertToNsUid(out nsUid, contentMetaId, "aoc"))
            {
                return false;
            }

            // Aoc 登録
            if (string.IsNullOrEmpty(nsUid))
            {
                // Aoc 新規登録
                var info = CreateAocInfo(applicationId, contentMetaId);
                var json = ToolUtility.Serialize<AocInfo>(info);
                return shop.RegisterContent(out nsUid, "aocs", json);
            }
            else
            {
                // Aoc 情報取得
                AocInfo info = null;
                if (!shop.GetAocInfo(out info, nsUid))
                {
                    return false;
                }

                // Aoc 更新登録
                UpdateAocInfo(info, applicationId, contentMetaId);
                var json = ToolUtility.Serialize<AocInfo>(info);
                return shop.UpdateContent("aocs", json, nsUid);
            }
        }
        // <-

        // Title 登録処理 ->
        private bool RegisterTitleByContentMeta(out string nsUid, Id64 applicationId, NsUidMap aocNsUids, bool isAppContained)
        {
            nsUid = string.Empty;

            var shop = new ShopHelper(Setting.Current);
            if (!shop.ConvertToNsUid(out nsUid, applicationId, "title"))
            {
                return false;
            }

            if (string.IsNullOrEmpty(nsUid))
            {
                // Title 新規登録
                var info = CreateTitleInfo(applicationId, aocNsUids, isAppContained);
                var json = ToolUtility.Serialize<TitleInfo>(info);
                if (!shop.RegisterContent(out nsUid, "titles", json))
                {
                    return false;
                }
            }
            else
            {
                // Title 情報取得
                TitleInfo info = null;
                if (!shop.GetTitleInfo(out info, nsUid))
                {
                    return false;
                }

                // Aoc 情報取得
                NsUidMap allNsUids = null;
                if (!shop.CreateTitleNsUidMap(out allNsUids, info, aocNsUids))
                {
                    return false;
                }

                // Title 更新登録
                UpdateTitleInfo(info, applicationId, aocNsUids, allNsUids, isAppContained);
                var json = ToolUtility.Serialize<TitleInfo>(info);
                if (!shop.UpdateContent("titles", json, nsUid))
                {
                    return false;
                }

                // Aoc 情報補足
                foreach (var key in allNsUids.Keys)
                {
                    if (!aocNsUids.ContainsKey(key))
                    {
                        aocNsUids.Add(key, allNsUids[key]);
                    }
                }
            }
            return true;
        }
        // <-

        private AocInfo CreateAocInfo(Id64 applicationId, Id64 contentMetaId)
        {
            var index = Id64.ToAocIndex(applicationId, contentMetaId);

            var info = new AocInfo();
            info.deviceType = "HAC";
            info.productClassCode = "HAC";
            info.formCode = "N";
            info.initialCode = Setting.Current.Naming.InitialCode;
            info.itemCodeOffset = index.ToString("000000");
            info.publisher = "NINTENDO";
            info.paymentRecipient = "NINTENDO";
            info.roms = new List<string> { contentMetaId.ToString() };
            info.dominantColorDefaultFlag = true;

            // 名称
            info.BuildNames(Setting.Current.Languages | TitleLanguagesRequired);
            foreach (var formaName in info.names.formalNames)
            {
                formaName.value = Setting.Current.Naming.ToFormalName(applicationId, formaName.lang, index);
            }

            // オンライン発売日
            info.onlineOnsaleDates = new List<AocInfo.OnlineOnsaleDate>();
            foreach (var country in Setting.Current.TargetCountries)
            {
                var onlineOnsaleDate = new AocInfo.OnlineOnsaleDate();
                onlineOnsaleDate.country = country;
                onlineOnsaleDate.year = 2017;
                onlineOnsaleDate.day = 1;
                onlineOnsaleDate.month = 1;
                info.onlineOnsaleDates.Add(onlineOnsaleDate);
            }

            // リリース
            info.releaseControls = new List<AocInfo.ReleaseControl>();
            foreach (var country in Setting.Current.TargetCountries)
            {
                var releaseControl = new AocInfo.ReleaseControl();
                releaseControl.country = country;
                releaseControl.publicStatus = "public";
                releaseControl.releaseDatetime = 3600000;
                info.releaseControls.Add(releaseControl);
            }

            // 販売
            info.salesControls = new AocInfo.SalesControls();
            info.salesControls.offDeviceSalesCountries = Setting.Current.TargetCountries;
            info.salesControls.onDeviceSalesCountries = Setting.Current.TargetCountries;
            return info;
        }

        private void UpdateAocInfo(AocInfo info, Id64 applicationId, Id64 contentMetaId)
        {
            var index = Id64.ToAocIndex(applicationId, contentMetaId);

            // 下記は必ず更新
            info.status = null;
            info.initialCode = Setting.Current.Naming.InitialCode;

            // Rom
            info.roms = new List<string> { contentMetaId.ToString() };

            // 名称
            info.UpdateNames(
                Setting.Current.Languages | TitleLanguagesRequired,
                (language) => { return Setting.Current.Naming.ToFormalName(applicationId, language, index); },
                Setting.Current.Naming.HasFormalName,
                Setting.Current.HasLanguages);
        }

        private TitleInfo CreateTitleInfo(Id64 applicationId, NsUidMap aocNsUids, bool isAppContained)
        {
            var index = 0;

            var info = new TitleInfo();
            info.applicationId = applicationId.ToString();
            info.deviceType = "HAC";
            info.productClassCode = "HAC";
            info.formCode = "P";
            info.initialCode = Setting.Current.Naming.InitialCode;
            info.platform = "HAC_DOWNLOADABLE";

            info.publisher = "NINTENDO";
            info.paymentRecipient = "NINTENDO";
            info.roms = new List<string> { applicationId.ToString() };
            info.dominantColorDefaultFlag = true;

            // ジャンル
            info.BuildGenre(Setting.Current.Languages | TitleLanguagesRequired);
            info.genre.codes.Add("ACTION");
            foreach (var name in info.genre.names)
            {
                name.value = "action";
            }

            // プレイヤー
            info.BuildPlayers();

            // 名称
            var naming = isAppContained ? Setting.Current.Naming : Setting.Current.Naming.CreateDefault();
            info.BuildNames(Setting.Current.Languages | TitleLanguagesRequired);
            foreach (var formaName in info.names.formalNames)
            {
                formaName.value = naming.ToFormalName(applicationId, formaName.lang, index);
            }

            // オンライン発売日
            info.onlineOnsaleDates = new List<TitleInfo.OnlineOnsaleDate>();
            foreach (var country in Setting.Current.TargetCountries)
            {
                var onlineOnsaleDate = new TitleInfo.OnlineOnsaleDate();
                onlineOnsaleDate.country = country;
                onlineOnsaleDate.year = 2017;
                onlineOnsaleDate.day = 1;
                onlineOnsaleDate.month = 1;
                info.onlineOnsaleDates.Add(onlineOnsaleDate);
            }

            // オリジナル発売日
            info.originalOnsaleDates = new List<TitleInfo.OriginalOnsaleDate>();
            foreach (var country in Setting.Current.TargetCountries)
            {
                var originalOnsaleDate = new TitleInfo.OriginalOnsaleDate();
                originalOnsaleDate.country = country;
                originalOnsaleDate.year = 2017;
                originalOnsaleDate.day = 1;
                originalOnsaleDate.month = 1;
                info.originalOnsaleDates.Add(originalOnsaleDate);
            }

            // パッケージ発売日
            info.packageOnsaleDates = new List<TitleInfo.PackageOnsaleDate>();
            foreach (var country in Setting.Current.TargetCountries)
            {
                var packageOnsaleDate = new TitleInfo.PackageOnsaleDate();
                packageOnsaleDate.country = country;
                packageOnsaleDate.year = 2017;
                packageOnsaleDate.day = 1;
                packageOnsaleDate.month = 1;
                info.packageOnsaleDates.Add(packageOnsaleDate);
            }

            // Aoc
            if (aocNsUids.Count > 0)
            {
                foreach (var country in Setting.Current.TargetCountries)
                {
                    info.BuildAoc(country, aocNsUids.ToNsUids(), false);
                }
            }

            // リリース
            info.releaseControls = new List<TitleInfo.ReleaseControl>();
            foreach (var country in Setting.Current.TargetCountries)
            {
                var releaseControl = new TitleInfo.ReleaseControl();
                releaseControl.country = country;
                releaseControl.publicStatus = "public";
                releaseControl.releaseDatetime = 3600000;
                info.releaseControls.Add(releaseControl);
            }

            // 販売
            info.salesControls = new TitleInfo.SalesControls();
            info.salesControls.packageSales = true;
            info.salesControls.downloadSales = true;
            info.salesControls.offDeviceSalesCountries = Setting.Current.TargetCountries;
            info.salesControls.onDeviceSalesCountries = Setting.Current.TargetCountries;
            info.salesControls.expectedOnlinePrices = new List<TitleInfo.SalesControls.ExpectedOnlinePrice>();
            foreach (var country in Setting.Current.TargetCountries)
            {
                var expectedOnlinePrice = new TitleInfo.SalesControls.ExpectedOnlinePrice();
                expectedOnlinePrice.country = country;
                expectedOnlinePrice.price = 0;
                info.salesControls.expectedOnlinePrices.Add(expectedOnlinePrice);
            }
            return info;
        }

        private void UpdateTitleInfo(TitleInfo info, Id64 applicationId, NsUidMap aocNsUids, NsUidMap allNsUids, bool isAppContained)
        {
            var index = 0;

            // 下記は必ず更新
            info.status = null;
            info.initialCode = Setting.Current.Naming.InitialCode;

            // Rom
            info.roms = new List<string> { applicationId.ToString() };

            // Aoc
            if (aocNsUids.Count > 0)
            {
                // 追加
                foreach (var country in Setting.Current.TargetCountries)
                {
                    info.BuildAoc(country, aocNsUids.ToNsUids(), false);
                }
            }
            if (info.aocs != null)
            {
                // ソート
                foreach (var aoc in info.aocs)
                {
                    var keys = new List<Id64>();
                    var map = new Dictionary<Id64, TitleInfo.Aoc.Id>();
                    if (aoc.ids != null)
                    {
                        foreach (var id in aoc.ids)
                        {
                            var contentMetaId = allNsUids.FindKey(id.id);
                            if (contentMetaId.IsValid)
                            {
                                keys.Add(contentMetaId);
                                map.Add(contentMetaId, id);
                            }
                            else
                            {
                                Log.WriteLine($"Warning: Unknown id. Not found Aoc of \"{id.id}\".");
                            }
                        }
                    }
                    if (keys.Count > 0)
                    {
                        keys.Sort();
                        aoc.ids = new List<TitleInfo.Aoc.Id>();
                        foreach (var key in keys)
                        {
                            aoc.ids.Add(map[key]);
                        }
                    }
                }
            }

            // プレイヤー（※ UI では未指定登録可能だが API からだと必須なので再構築する）
            info.BuildPlayers();

            // 名称（※アプリを含めて指定された場合のみ変更）
            if (isAppContained)
            {
                info.UpdateNames(
                    Setting.Current.Languages | TitleLanguagesRequired,
                    (language) => { return Setting.Current.Naming.ToFormalName(applicationId, language, index); },
                    Setting.Current.Naming.HasFormalName,
                    Setting.Current.HasLanguages);
            }
        }

        private void WriteNsUidList(string nsUid, Id64 applicationId, NsUidMap aocNsUids)
        {
            // リスト出力
            Log.WriteLine($"========================================");
            Log.WriteLine($"NsUids:");
            Log.WriteLine($"  ApplicationNsUid:  {nsUid} ({applicationId})");

            var keys = new List<Id64>(aocNsUids.Keys);
            keys.Sort();
            foreach (var key in keys)
            {
                Log.WriteLine($"  AddOnContentNsUid: {aocNsUids[key]} ({key})");
            }
        }
    }
}
