﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using Nintendo.Foundation.IO;
using ContentsUploader.Assistants;

namespace ContentsUploader.Commands
{
    using static Constants;
    using static Models.Pms;

    public class RegisterDemoCommand : CommandBase
    {
        //! nsp 指定
        [CommandLineOption('s', "input", DefaultValue = "", Description = "The file/derctory path of the nsp that you want to register-demo.")]
        public override string NspPathOption { get; set; }

        //! 単体 ID 指定
        [CommandLineOption("application-id", DefaultValue = "", Description = "input application id")]
        public string ApplicationIdNewOption { get; set; }

        //! 単体 ID 指定 (旧オプション)
        [CommandLineOption('a', "application", DefaultValue = "", IsHidden = true, Description = "input application id")]
        public string ApplicationIdOldOption { get; set; }

        [CommandLineOption("name", DefaultValue = "", Description = "item name")]
        public string FormalNameOption { get; set; }

        [CommandLineOption("language", DefaultValue = "", Description = "item language")]
        public string LanguageOption { get; set; }

        public override string ApplicationIdOption
        {
            get { return !string.IsNullOrEmpty(ApplicationIdNewOption) ? ApplicationIdNewOption : ApplicationIdOldOption; }
            set { ApplicationIdNewOption = value; }
        }

        public override void Run()
        {
            Run("Register Demo");
        }

        protected override bool ValidateOptions()
        {
            var name = Naming.CreateDemo(FormalNameOption);
            var valid = name.IsValid;
            valid &= Setting.Current.SetupNaming(name);
            valid &= Setting.Current.SetupDemoLanguages(LanguageOption);
            return valid;
        }

        //! 実行処理 ->
        protected override bool RunByDirectory()
        {
            var path = string.Empty;
            if (!ToolUtility.ConvertToAbsoluteDirectoryPath(out path, NspPathOption))
            {
                Log.WriteLine($"Error: Directory not found. Please check the path \"{path}\".");
                return false;
            }

            // Rom 登録
            var shop = new ShopHelper(Setting.Current);
            var result = new RegisterRomResult();
            if (!shop.RegisterRomByDirectory(out result, path))
            {
                return false;
            }

            // Demo 登録
            if (result.IsPatchOnly())
            {
                // Patch のみの場合、以降の Demo API は使わない
                return true;
            }
            return RegisterDemoByContentMeta(result.ApplicationId, result.ToAppsAndAocsList());
        }

        protected override bool RunByFile()
        {
            var path = string.Empty;
            if (!ToolUtility.ConvertToAbsoluteNspFilePath(out path, NspPathOption))
            {
                Log.WriteLine($"Error: Nsp file not found. Please check the path \"{path}\".");
                return false;
            }

            // Rom 登録
            var shop = new ShopHelper(Setting.Current);
            var result = new RegisterRomResult();
            if (!shop.RegisterRomByFile(out result, path))
            {
                return false;
            }

            // Demo 登録
            if (result.IsPatchOnly())
            {
                // Patch のみの場合、以降の Demo API は使わない
                return true;
            }
            return RegisterDemoByContentMeta(result.ApplicationId, result.ToAppsAndAocsList());
        }

        protected override bool RunByApplicationId(Id64 applicationId)
        {
            // Rom 登録
            var shop = new ShopHelper(Setting.Current);
            if (!shop.RegisterRomByContentMeta(applicationId, ContentMetaType.Application))
            {
                return false;
            }

            // Demo 登録
            var roms = new List<Id64>() { applicationId };
            return RegisterDemoByContentMeta(applicationId, roms);
        }
        //! <-

        // Demo 登録処理 ->
        private bool RegisterDemoByContentMeta(Id64 applicationId, List<Id64> contentMetaIds)
        {
            var nsUid = "";

            // 対象コンテンツ Demo の nsUid を特定
            // 注意、Demo 単位で nsUid を持つので、下記手順なら nsUid を新規に取得する
            //  1. App+AoC 登録 ⇒ nsUid:1 に新規登録
            //  2. AoC のみ登録 ⇒ nsUid:1 を更新登録、App は nsUid:1 の対象から外れる
            //  3. App のみ登録 ⇒ nsUid:1 の対象ではないので nsUid:2 に新規登録
            var shop = new ShopHelper(Setting.Current);
            if (!shop.ConvertToNsUid(out nsUid, contentMetaIds[0], "demo"))
            {
                return false;
            }

            // Demo 登録
            if (string.IsNullOrEmpty(nsUid))
            {
                // Demo 新規登録
                var info = CreateDemoInfo(applicationId, contentMetaIds);
                var json = ToolUtility.Serialize<DemoInfo>(info);
                if (!shop.RegisterContent(out nsUid, "demos", json))
                {
                    return false;
                }
            }
            else
            {
                // Demo 情報取得
                DemoInfo info;
                if (!shop.GetDemoInfo(out info, nsUid))
                {
                    return false;
                }

                // Demo 更新登録
                UpdateDemoInfo(info, applicationId, contentMetaIds);
                var json = ToolUtility.Serialize<DemoInfo>(info);
                if (!shop.UpdateContent("demos", json, nsUid))
                {
                    return false;
                }
            }

            // リスト出力
            Log.WriteLine($"========================================");
            Log.WriteLine($"NsUids:");
            Log.WriteLine($"  DemoNsUid: {nsUid} ({string.Join(",", contentMetaIds)})");
            return true;
        }
        // <-

        private DemoInfo CreateDemoInfo(Id64 applicationId, List<Id64> contentMetaIds)
        {
            var index = Id64.ToAocIndex(applicationId, contentMetaIds[0]);

            var info = new DemoInfo();
            info.deviceType = "HAC";
            info.productClassCode = "HAC";
            info.formCode = "U";
            info.initialCode = Setting.Current.Naming.InitialCode;
            info.platform = "HAC_DOWNLOADABLE";
            info.publisher = "NINTENDO";

            // レーティング
            info.ratings = new List<DemoInfo.Rating>();
            {
                var rating = new DemoInfo.Rating();
                rating.system = "NONE";
                rating.id = 0;
                rating.targetCountries = DemoCountries;
                info.ratings.Add(rating);
            }

            // Rom
            info.roms = new List<string>();
            foreach (var contentMetaId in contentMetaIds)
            {
                info.roms.Add(contentMetaId.ToString());
            }

            // 名称
            info.BuildNames(Setting.Current.Languages | DemoLanguagesRequired);
            foreach (var formaName in info.names.formalNames)
            {
                formaName.value = Setting.Current.Naming.ToFormalName(applicationId, formaName.lang, index);
            }

            // リリース
            info.releaseControls = new List<DemoInfo.ReleaseControl>();
            foreach (var country in DemoCountries)
            {
                var releaseControl = new DemoInfo.ReleaseControl();
                releaseControl.country = country;
                releaseControl.publicStatus = "public";
                releaseControl.releaseDatetime = 3600000;
                info.releaseControls.Add(releaseControl);
            }
            return info;
        }

        private void UpdateDemoInfo(DemoInfo info, Id64 applicationId, List<Id64> contentMetaIds)
        {
            var index = Id64.ToAocIndex(applicationId, contentMetaIds[0]);

            // 下記は必ず更新
            info.platform = "HAC_DOWNLOADABLE";
            info.status = null;

            // Rom
            info.roms = new List<string>();
            foreach (var contentMetaId in contentMetaIds)
            {
                info.roms.Add(contentMetaId.ToString());
            }

            // 名称
            info.UpdateNames(
                Setting.Current.Languages | DemoLanguagesRequired,
                (language) => { return Setting.Current.Naming.ToFormalName(applicationId, language, index); },
                Setting.Current.Naming.HasFormalName,
                Setting.Current.HasLanguages);
        }
    }
}
