﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using Nintendo.Foundation.IO;
using ContentsUploader.Assistants;

namespace ContentsUploader.Commands
{
    using static Constants;
    using static Models.Pms;

    public class RegisterBundleCommand : PricingCommand
    {
        [CommandLineOption("ns-uid", DefaultValue = "", Description = "ns uid.", IsRequired = true)]
        public string NsUidOption { get; set; }

        [CommandLineOption('s', "input", DefaultValue = "", Description = "The file/derctory path of the nsp that you want to register-bundle.")]
        public override string NspPathOption { get; set; }

        [CommandLineOption("bundle-id", DefaultValue = "", Description = "Bundle content id.")]
        public string BundleIdOption { get; set; }

        // バンドルの操作指定
        enum NsUidState
        {
            Invalid,    // 無効
            New,        // 新規
            Exist,      // 既存
        };
        private NsUidState BundleState { get; set; } = NsUidState.Invalid;
        private List<Id64> BundleIds { get; } = new List<Id64>();

        protected override bool HasOtherOption()
        {
            return !string.IsNullOrEmpty(BundleIdOption) || string.IsNullOrEmpty(NspPathOption);
        }

        public override void Run()
        {
            Run("Register Bundle", Mode.IsTokenRequired);
        }

        protected override bool ValidateOptions()
        {
            var name = Naming.CreateTitle(InitialCodeOption, FormalNameOption);
            var valid = name.IsValid;
            valid &= Setting.Current.SetupNaming(name);
            valid &= Setting.Current.SetupTitleLanguages(LanguageOption);
            valid &= Setting.Current.SetupTitleCountries(TargetCountryOption);
            valid &= SetupSaleOperations();
            valid &= SetupBundleTarget();
            return valid;
        }

        // バンドル引数の整合性確認
        private bool SetupBundleTarget()
        {
            var valid = true;
            try
            {
                // バンドル ns-uid の状態
                if (string.IsNullOrEmpty(NsUidOption))
                {
                    Log.WriteLine($"Error: Invalid ns uid.");
                    valid = false;
                }
                else if (string.Compare(NsUidOption, "new") == 0)
                {
                    BundleState = NsUidState.New;
                }
                else
                {
                    BundleState = NsUidState.Exist;
                }

                // 新規ならバンドル対象のコンテンツ指定は必須
                if (BundleState == NsUidState.New
                    && string.IsNullOrEmpty(BundleIdOption)
                    && string.IsNullOrEmpty(NspPathOption))
                {
                    Log.WriteLine($"Error: Please input bundle contents.");
                    valid = false;
                }

                // バンドル対象のコンテンツの id をパース
                var delimiter = new char[] { ',' };
                var values = BundleIdOption.Split(delimiter, StringSplitOptions.RemoveEmptyEntries);
                foreach (var value in values)
                {
                    BundleIds.Add(new Id64(value));
                }
            }
            catch (Exception)
            {
                Log.WriteLine($"Error: Invalid bundle id.");
                valid = false;
            }
            return valid;
        }

        //! 実行処理 ->
        protected override bool RunByDirectory()
        {
            var path = string.Empty;
            if (!ToolUtility.ConvertToAbsoluteDirectoryPath(out path, NspPathOption))
            {
                Log.WriteLine($"Error: Directory not found. Please check the path \"{path}\".");
                return false;
            }

            // Rom 確認
            var shop = new ShopHelper(Setting.Current);
            var bundleIds = new RomCollection();
            if (!shop.ConfirmRomByDirectory(out bundleIds, path))
            {
                return false;
            }

            // Bundle 登録
            var nsUid = string.Empty;
            if (!RegisterBundle(out nsUid, bundleIds))
            {
                return false;
            }
            WriteNsUidList(nsUid);
            return true;
        }

        protected override bool RunByFile()
        {
            var path = string.Empty;
            if (!ToolUtility.ConvertToAbsoluteNspFilePath(out path, NspPathOption))
            {
                Log.WriteLine($"Error: Nsp file not found. Please check the path \"{path}\".");
                return false;
            }

            // Rom 確認
            var shop = new ShopHelper(Setting.Current);
            var bundleIds = new RomCollection();
            if (!shop.ConfirmRomByFile(out bundleIds, path))
            {
                return false;
            }

            // Bundle 登録
            var nsUid = string.Empty;
            if (!RegisterBundle(out nsUid, bundleIds))
            {
                return false;
            }
            WriteNsUidList(nsUid);
            return true;
        }

        protected override bool RunByOther()
        {
            // Rom 確認
            var shop = new ShopHelper(Setting.Current);
            var bundleIds = new RomCollection();
            foreach (var contentMetaId in BundleIds)
            {
                ContentMetaType contentMetaType;
                if (!shop.ConfirmRomByContentMeta(out contentMetaType, contentMetaId))
                {
                    return false;
                }
                bundleIds.Add(contentMetaId, contentMetaType);
            }

            // Bundle 登録
            var nsUid = string.Empty;
            if (!RegisterBundle(out nsUid, bundleIds))
            {
                return false;
            }
            WriteNsUidList(nsUid);
            return true;
        }

        private void WriteNsUidList(string nsUid)
        {
            // リスト出力
            Log.WriteLine($"========================================");
            Log.WriteLine($"NsUids:");
            Log.WriteLine($"  BundleNsUid: {nsUid}");
        }
        //! <-

        // Bundle 登録処理 ->
        private bool RegisterBundle(out string nsUid, RomCollection bundleIds)
        {
            nsUid = string.Empty;

            // バンドル対象のコンテンツ数が十分か確認
            // ・コンテンツを指定する場合は、コンテンツを二つ以上指定する必要がある
            // ・更新登録なら、登録済みの内容で更新するので、コンテンツの指定はなくてもかまわない
            var bundleCount = bundleIds.Apps.Count + bundleIds.Aocs.Count;
            var isEnough = (bundleCount >= 2) || (BundleState == NsUidState.Exist && bundleCount == 0);
            if (!isEnough)
            {
                Log.WriteLine($"Error: Please specify two or more application/aoc contents.");
                return false;
            }

            // Title,Aoc 確認
            var shop = new ShopHelper(Setting.Current);
            NsUidMap bundleNsUids = null;
            if (!shop.ConfrimContents(out bundleNsUids, bundleIds))
            {
                return false;
            }

            // Bundle 登録
            if (BundleState == NsUidState.New)
            {
                // Bundle 新規登録
                var info = CreateBundleInfo(bundleIds, bundleNsUids);
                var json = ToolUtility.Serialize<BundleInfo>(info);
                if (!shop.RegisterContent(out nsUid, "bundles", json))
                {
                    return false;
                }
            }
            else
            {
                // Bundle 情報取得
                nsUid = NsUidOption;
                BundleInfo info = null;
                if (!shop.GetBundleInfo(out info, nsUid))
                {
                    return false;
                }

                // Bundle 更新登録
                UpdateTitleInfo(info, bundleIds, bundleNsUids);
                var json = ToolUtility.Serialize<BundleInfo>(info);
                if (!shop.UpdateContent("bundles", json, nsUid))
                {
                    return false;
                }
            }

            // Price 登録
            if (!RegisterPriceByNsUid("bundle", nsUid, true, true))
            {
                return false;
            }
            return true;
        }

        private BundleInfo CreateBundleInfo(RomCollection bundleIds, NsUidMap bundleNsUids)
        {
            var applicationId = Id64.Invalid;
            var index = 0;

            var info = new BundleInfo();
            info.deviceType = "HAC";
            info.productClassCode = "HAC";
            info.formCode = "A";
            info.initialCode = Setting.Current.Naming.InitialCode;

            info.publisher = "NINTENDO";
            info.paymentRecipient = "NINTENDO";
            info.dominantColorDefaultFlag = true;

            // コンテンツ
            SetBundleType(info, bundleIds);
            info.contents = new List<BundleInfo.Content>();
            foreach (var contentMetaId in bundleIds.Apps)
            {
                var content = new BundleInfo.Content();
                content.id = long.Parse(bundleNsUids[contentMetaId]);
                content.type = "TITLE";
                info.contents.Add(content);
            }
            foreach (var contentMetaId in bundleIds.Aocs)
            {
                var content = new BundleInfo.Content();
                content.id = long.Parse(bundleNsUids[contentMetaId]);
                content.type = "AOC";
                info.contents.Add(content);
            }

            // ジャンル
            info.genre = new BundleInfo.Genre();
            info.genre.codes = new List<string>();
            info.genre.codes.Add("ACTION");

            // 名称
            var naming = Setting.Current.Naming;
            info.BuildNames(Setting.Current.Languages | TitleLanguagesRequired);
            foreach (var formaName in info.names.formalNames)
            {
                formaName.value = naming.ToFormalName(applicationId, formaName.lang, index);
            }

            // オンライン発売日
            info.onlineOnsaleDates = new List<BundleInfo.OnlineOnsaleDate>();
            foreach (var country in Setting.Current.TargetCountries)
            {
                var onlineOnsaleDate = new BundleInfo.OnlineOnsaleDate();
                onlineOnsaleDate.country = country;
                onlineOnsaleDate.year = 2017;
                onlineOnsaleDate.day = 1;
                onlineOnsaleDate.month = 1;
                info.onlineOnsaleDates.Add(onlineOnsaleDate);
            }

            // リリース
            info.releaseControls = new List<BundleInfo.ReleaseControl>();
            foreach (var country in Setting.Current.TargetCountries)
            {
                var releaseControl = new BundleInfo.ReleaseControl();
                releaseControl.country = country;
                releaseControl.publicStatus = "public";
                releaseControl.releaseDatetime = 3600000;
                info.releaseControls.Add(releaseControl);
            }

            // 販売
            info.salesControls = new BundleInfo.SalesControls();
            info.salesControls.offDeviceSalesCountries = Setting.Current.TargetCountries;
            info.salesControls.onDeviceSalesCountries = Setting.Current.TargetCountries;
            info.salesControls.expectedOnlinePrices = new List<BundleInfo.SalesControls.ExpectedOnlinePrice>();
            foreach (var country in Setting.Current.TargetCountries)
            {
                var expectedOnlinePrice = new BundleInfo.SalesControls.ExpectedOnlinePrice();
                expectedOnlinePrice.country = country;
                expectedOnlinePrice.price = 0;
                info.salesControls.expectedOnlinePrices.Add(expectedOnlinePrice);
            }
            return info;
        }

        private void UpdateTitleInfo(BundleInfo info, RomCollection bundleIds, NsUidMap bundleNsUids)
        {
            var applicationId = Id64.Invalid;
            var index = 0;

            // 下記は必ず更新
            info.status = null;
            info.initialCode = Setting.Current.Naming.InitialCode;

            // コンテンツ更新
            if ((bundleIds.Apps.Count > 0) || (bundleIds.Aocs.Count > 0))
            {
                SetBundleType(info, bundleIds);
                info.contents = new List<BundleInfo.Content>();
                foreach (var contentMetaId in bundleIds.Apps)
                {
                    var content = new BundleInfo.Content();
                    content.id = long.Parse(bundleNsUids[contentMetaId]);
                    content.type = "TITLE";
                    info.contents.Add(content);
                }
                foreach (var contentMetaId in bundleIds.Aocs)
                {
                    var content = new BundleInfo.Content();
                    content.id = long.Parse(bundleNsUids[contentMetaId]);
                    content.type = "AOC";
                    info.contents.Add(content);
                }
            }

            // 名称更新
            info.UpdateNames(
                Setting.Current.Languages | TitleLanguagesRequired,
                (language) => { return Setting.Current.Naming.ToFormalName(applicationId, language, index); },
                Setting.Current.Naming.HasFormalName,
                Setting.Current.HasLanguages);
        }

        private void SetBundleType(BundleInfo info, RomCollection bundleIds)
        {
            if (bundleIds.IsAppOnly())
            {
                info.bundleType = "title";
                info.itemCodeOffset = null;
            }
            else if (bundleIds.IsAocOnly())
            {
                info.bundleType = "aoc";
                info.itemCodeOffset = "000001";
            }
            else if ((bundleIds.Apps.Count > 0) && (bundleIds.Aocs.Count > 0))
            {
                info.bundleType = "title_aoc";
                info.itemCodeOffset = "000001";
            }
            else
            {
                info.bundleType = null;
                info.itemCodeOffset = null;
            }
        }
        //! <-
    }
}
