﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.Serialization.Json;
using System.Text;
using System.Text.RegularExpressions;

namespace ContentsUploader.Assistants
{
    public static class ToolUtility
    {
        public static bool GetTargetJsonValue(out string targetValue, string jsonText, string target)
        {
            // TODO: string 型以外の値も取得できるようにする
            string[] inners = jsonText.Split(new char[] { '\"' });
            int pos = 0;
            bool isTarget = false;
            foreach (var value in inners)
            {
                if (value == target)
                {
                    isTarget = true;
                    break;
                }
                pos++;
            }

            if (isTarget)
            {
                targetValue = inners[pos + 2];
                return true;
            }
            else
            {
                targetValue = string.Empty;
                return false;
            }
        }

        // json をインデントした書式で変換する
        public static string FormatIndentedJson(string jsonText)
        {
            var sb = new StringBuilder();

            var spaces = "";    // インデント分の空白
            int indent = 0;     // インデント数
            int quote = 0;      // ダブルクォート数
            int newline = -1;   // 0 以上なら値分の文字数を含めて改行
            int lastline = 0;   // 出力した最終行の位置

            var str = jsonText.Trim();
            var prv = '\0';
            var cur = '\0';
            for (int index = 0; index < str.Length; index++)
            {
                prv = cur;
                cur = str[index];

                // ダブルクォートのカウント、但し \" は文字なので除外
                if (prv != '\\' && cur == '"')
                {
                    quote++;
                }

                // "～" の外側なら下記条件
                if ((quote % 2) == 0)
                {
                    if (cur == ',')
                    {
                        // , を含めて改行指定
                        newline = 1;
                    }
                    else if (cur == '{' || cur == '[')
                    {
                        // { または [ を含めて改行指定
                        newline = 1;
                        indent++;
                    }
                    else if (cur == '}' || cur == ']')
                    {
                        // } または ] を含めずに改行指定（※次行に含む）
                        newline = 0;
                        indent--;
                    }

                    if (newline >= 0)
                    {
                        // 一行分追加
                        var sub = str.Substring(lastline, index + newline - lastline).Trim();
                        if (!string.IsNullOrEmpty(sub))
                        {
                            sb.Append(spaces);
                            sb.AppendLine(sub);
                        }

                        // 次行のインデントを準備
                        spaces = new string(' ', indent * 2);

                        // 出力した最終行の位置を更新、改行指定を解除
                        lastline = index + newline;
                        newline = -1;
                    }
                }
            }
            {
                var sub = str.Substring(lastline).Trim();
                if (!string.IsNullOrEmpty(sub))
                {
                    sb.Append(spaces);
                    sb.Append(str.Substring(lastline));
                }
            }
            return sb.ToString();
        }

        // json の指定データの値を置換して返す
        public static string ReplaceJsonValue(string jsonText, string dataName, bool newValue)
        {
            var prefix = $"\"{dataName}\":";
            var src = prefix + (!newValue).ToString().ToLowerInvariant();
            var dst = prefix + (newValue).ToString().ToLowerInvariant();
            return jsonText.Replace(src, dst);
        }

        // 絶対パス変換、対象ファイルが存在すれば真を返す
        public static bool ConvertToAbsoluteFilePath(out string full, string path)
        {
            var info = new FileInfo(path);
            full = info.FullName;
            return info.Exists;
        }

        // 絶対パス変換、対象 NSP ファイルが存在すれば真を返す
        public static bool ConvertToAbsoluteNspFilePath(out string full, string path)
        {
            var info = new FileInfo(path);
            full = info.FullName;
            return info.Exists && (string.Compare(info.Extension, ".nsp", true) == 0);
        }

        // 絶対パス変換、対象ディレクトリが存在すれば真を返す
        public static bool ConvertToAbsoluteDirectoryPath(out string full, string path)
        {
            var info = new DirectoryInfo(path);
            full = info.FullName;
            return info.Exists;
        }

        // デシリアイズ
        public static T Deserialize<T>(string json)
        {
            var serializer = new DataContractJsonSerializer(typeof(T));
            using (var stream = new MemoryStream(Encoding.UTF8.GetBytes(json)))
            {
                return (T)serializer.ReadObject(stream);
            }
        }

        // シリアライズ
        public static string Serialize<T>(T data)
        {
            var serializer = new DataContractJsonSerializer(typeof(T));
            using (var stream = new MemoryStream())
            {
                serializer.WriteObject(stream, data);
                return Encoding.UTF8.GetString(stream.ToArray());
            }
        }

        // null 比較、null の方が負、両方とも null なら正、null があれば真を返す
        public static bool CompareNull<T>(out int result, T lhs, T rhs)
        {
            if (object.ReferenceEquals(lhs, null))
            {
                result = object.ReferenceEquals(rhs, null) ? 0 : -1;
                return true;
            }
            else if (object.ReferenceEquals(rhs, null))
            {
                result = 1;
                return true;
            }
            result = 0;
            return false;
        }

        // 二つのオブジェクトを比較する、同値でなければ真の返す
        public static bool Compare<T>(out int result, T lhs, T rhs) where T : IComparable
        {
            if (object.ReferenceEquals(lhs, null))
            {
                result = object.ReferenceEquals(rhs, null) ? 0 : -1;
            }
            else if (object.ReferenceEquals(rhs, null))
            {
                result = 1;
            }
            else
            {
                result = lhs.CompareTo(rhs);
            }
            return result != 0;
        }

        // 二つのオブジェクトを比較する
        public static int Compare<T>(T lhs, T rhs) where T : IComparable
        {
            int result = 0;
            Compare<T>(out result, lhs, rhs);
            return result;
        }

        // 与えられた関数で比較する
        public static int Compare<T>(T lhs, T rhs, Comparison<T> comparison)
        {
            if (object.ReferenceEquals(lhs, null))
            {
                return object.ReferenceEquals(rhs, null) ? 0 : -1;
            }
            if (object.ReferenceEquals(rhs, null))
            {
                return 1;
            }
            return comparison(lhs, rhs);
        }

        // Enum 値 -> 定義名配列
        public static string[] ToEnumNames<T>(T value) where T : struct
        {
            var delimiter = new char[] { ',', ' ' };
            return value.ToString().Split(delimiter, StringSplitOptions.RemoveEmptyEntries);
        }

        // 文字列 -> Enum 値変換
        public static T ToEnumValue<T>(string value, T @default) where T : struct
        {
            T result;
            if (!Enum.TryParse(value, out result))
            {
                result = @default;
            }
            return result;
        }

        // 文字列 -> Enum 値変換（変換失敗時は例外）
        public static T ToEnumValue<T>(string value) where T : struct
        {
            return (T)Enum.Parse(typeof(T), value);
        }

        // 文字列 -> int 型配列変換
        public static int[] ToIntArray(string values, string delimiters = ", ")
        {
            var list = new List<int>();
            foreach (var v in values.Trim().Split(delimiters.ToCharArray()))
            {
                list.Add(int.Parse(v));
            }
            return list.ToArray();
        }

        // 文字列 -> 日付型変換
        public static DateTime ToDateTime(string value)
        {
            return DateTime.ParseExact(
                value, "yyyy/M/d H:m",
                System.Globalization.CultureInfo.InvariantCulture,
                System.Globalization.DateTimeStyles.None);
        }

        // 指定時間を含む区間の開始日時（00:00 〜）
        public static DateTime ToTimePeriodBegin(DateTime value)
        {
            const long TimePeriodSeconds = 300;     // 時間は 300 秒毎に区切る
            const long OneSecondTick = 10000000;    // 1 秒は 10,000,000 tick
            const long TimePeriodTicks = TimePeriodSeconds * OneSecondTick;

            var num = (long)(value.Ticks / TimePeriodTicks);
            return new DateTime(num * TimePeriodTicks);
        }

        // 指定時間を含む区間の終了日時（〜 09:59）
        public static DateTime ToTimePeriodEnd(DateTime value)
        {
            var next = ToTimePeriodBegin(value.AddMinutes(10));
            return next.AddSeconds(-1);
        }

        // 指定時間を含まない次区間の開始日時（10:00 〜）
        public static DateTime ToTimePeriodNext(DateTime value)
        {
            var next = ToTimePeriodBegin(value.AddMinutes(10));
            return next;
        }

        // 日付型 -> ISO-8601 形式文字列
        public static string ToIso8601Utc(DateTime value)
        {
            return TimeZoneInfo.ConvertTimeToUtc(value).ToString("yyyy-MM-ddTHH:mm:ss+0000");
        }

        // 指定範囲（下限～上限）に制限
        public static T Clamp<T>(T val, T min, T max) where T : IComparable
        {
            //return (val < min) ? min : ((val > max) ? max : val);
            return (val.CompareTo(min) < 0) ? min : ((val.CompareTo(max) > 0) ? max : val);
        }

        // 指定処理が成功するまで間隔空けて試行回数繰り返す
        public static bool RetryUntilSuccess(Func<bool> function, int retry, int interval)
        {
            for (var count = 0; count < retry; count++)
            {
                if (count > 0)
                {
                    System.Threading.Thread.Sleep(interval);
                }
                if (function())
                {
                    return true;
                }
            }
            return false;
        }

        // イニシャルコードとして利用可能かを検証する
        public static bool ValidateInitialCode(string code)
        {
            if (!string.IsNullOrEmpty(code) && Regex.IsMatch(code, @"^[0-9a-zA-Z]{5}$"))
            {
                return true;
            }
            return false;
        }
    }
}
